-- wallgoo/portal.lua

wallgoo = wallgoo or {}

local portal_nodes = {
    frame = "wallygoo2:portal_frame",
    core = "wallygoo2:portal_core",
    active = "wallygoo2:portal_active"
}

-- Register portal frame
minetest.register_node(portal_nodes.frame, {
    description = "Wallgoo Portal Frame",
    tiles = {"wallgoo_portal_frame.png"},
    groups = {cracky = 2},
})

-- Register portal core (activator)
minetest.register_node(portal_nodes.core, {
    description = "Wallgoo Portal Core",
    tiles = {"wallgoo_portal_core.png"},
    groups = {cracky = 1},
    on_rightclick = function(pos, node, clicker, itemstack)
        wallgoo.activate_portal(pos, clicker)
    end,
})

-- Register active portal node
minetest.register_node(portal_nodes.active, {
    description = "Wallgoo Portal",
    tiles = {"wallgoo_portal_active.png"},
    light_source = 10,
    walkable = false,
    pointable = false,
    diggable = false,
    buildable_to = true,
    groups = {not_in_creative_inventory = 1},
    drop = "",
})

-- Portal activation logic
function wallgoo.activate_portal(core_pos, clicker)
    local function is_frame(pos)
        return minetest.get_node(pos).name == "wallygoo2:portal_frame"
    end

    local offsets = {
        {x=-1, y=1, z=0}, {x=0, y=1, z=0}, {x=1, y=1, z=0},
        {x=-1, y=0, z=0},                 {x=1, y=0, z=0},
        {x=-1, y=-1, z=0}, {x=0, y=-1, z=0}, {x=1, y=-1, z=0},
    }

    for _, offset in ipairs(offsets) do
        local check_pos = vector.add(core_pos, offset)
        if not is_frame(check_pos) then
            minetest.chat_send_player(clicker:get_player_name(), "Portal frame incomplete!")
            return
        end
    end

    minetest.set_node(core_pos, {name = "wallygoo2:portal_active"})
    minetest.chat_send_player(clicker:get_player_name(), "Portal activated!")
end

-- Teleport logic
minetest.register_globalstep(function(dtime)
    for _, player in ipairs(minetest.get_connected_players()) do
        local pos = vector.round(player:get_pos())
        local node = minetest.get_node(pos)
        if node.name == "wallygoo2:portal_active" then
            wallgoo.teleport_player(player)
        end
    end
end)

function wallgoo.teleport_player(player)
    local name = player:get_player_name()
    local target_pos = {x=1000, y=10, z=1000} -- Wallgoo dimension spawn
    minetest.chat_send_player(name, "🌌 Entering the Wallgoo Dimension...")
    player:set_pos(target_pos)
end
minetest.register_on_generated(function(minp, maxp, seed)
    -- Only generate in the Wallgoo dimension
    if minp.x > 900 and minp.x < 1100 and minp.z > 900 and minp.z < 1100 then
        local vm, emin, emax = minetest.get_mapgen_object("voxelmanip")
        local area = VoxelArea:new{MinEdge=emin, MaxEdge=emax}
        local data = vm:get_data()

        local c_wallgoo = minetest.get_content_id("wallygoo2:basic_block")
        local c_air = minetest.get_content_id("air")

        for z = minp.z, maxp.z do
            for y = minp.y, maxp.y do
                for x = minp.x, maxp.x do
                    local vi = area:index(x, y, z)
                    if y < 10 then
                        data[vi] = c_wallgoo
                    else
                        data[vi] = c_air
                    end
                end
            end
        end

        vm:set_data(data)
        vm:write_to_map()
        vm:update_map()
    end
end)
minetest.register_node("wallygoo2:return_portal_core", {
    description = "Wallgoo Return Portal",
    tiles = {"wallgoo_return_portal_core.png"},
    groups = {cracky = 1},
    light_source = 10,
    on_rightclick = function(pos, node, clicker, itemstack)
        wallgoo.return_to_overworld(clicker)
    end,
})
function wallgoo.return_to_overworld(player)
    local name = player:get_player_name()
    local target_pos = {x=0, y=10, z=0} -- Overworld spawn or safe return point
    minetest.chat_send_player(name, "🌀 Returning to the Overworld...")
    player:set_pos(target_pos)
end
minetest.register_craft({
    type = "shapeless",
    output = "wallygoo2:portal_frame",
    recipe = {"wallygoo2:shrine_research_template",
        "wallygoo2:goo",
        "wallygoo2:basic_block",
    }
})

minetest.log("action", "[Wallgoo] Registered portal frame recipe")


minetest.register_craft({
    output = "wallygoo2:portal_core",
    recipe = {
        {"wallygoo2:brick", "wallygoo2:shrine_research_template", "wallygoo2:brick"},
        {"wallygoo2:brick", "default:mese_crystal", "wallygoo2:brick"},
    }
})
minetest.register_craft({
    output = "wallygoo2:return_portal_core",
    recipe = {
        {"", "wallygoo2:portal_core", ""},
        {"", "wallygoo2:goo", ""},
    }
})

