--========================
-- 🛕 Shrine Spawner Node
--========================

minetest.register_node("wallygoo2:shrine_spawner", {
    description = "Wallgoo Shrine Spawner",
    tiles = {
        "wallgoo_spawner_top_bottom.png",  -- top
        "wallgoo_spawner_top_bottom.png",  -- bottom
        "wallgoo_spawn_side.png",          -- right
        "wallgoo_spawn_side.png",          -- left
        "wallgoo_spawn_side.png",          -- back
        "wallgoo_spawn_side.png",          -- front
    },
    groups = {cracky = 1, oddly_breakable_by_hand = 2},
    sounds = default.node_sound_stone_defaults(),

    on_construct = function(pos)
        local player = minetest.get_player_by_name("singleplayer")
        local name = player and player:get_player_name() or "unknown"
        minetest.chat_send_player(name, "Shrine spawner placed at " .. minetest.pos_to_string(pos))
    end,

    after_place_node = function(pos, placer)
        local shrine_pos = vector.add(pos, {x = 0, y = 1, z = 0})
        place_wallgoo_shrine(shrine_pos)
        minetest.chat_send_player(placer:get_player_name(), "Shrine placed above spawner.")
    end,
})

--========================
-- 🛕 Shrine Placement Function (X-shaped layout)
--========================

function place_wallgoo_shrine(pos)
    if not pos then
        minetest.log("error", "[Wallgoo2] Shrine placement failed: nil position")
        return
    end

    local function set(dx, dy, dz, nodename)
        local target = vector.add(pos, {x = dx, y = dy, z = dz})
        if minetest.get_node(target).name == "air" then
            minetest.set_node(target, {name = nodename})
        else
            minetest.log("warning", "[Wallgoo2] Blocked at " .. minetest.pos_to_string(target))
        end
    end

    -- Layer 0: Base
    for x = -2, 2 do
        for z = -2, 2 do
            set(x, 0, z, "wallygoo2:brick")
        end
    end

    -- Layer 1: X-shaped shrine/glass, rest bricks
    for x = -2, 2 do
        for z = -2, 2 do
            if x == z or x == -z then
                set(x, 1, z, (x == 0 and z == 0) and "wallygoo2:shrine" or "wallygoo2:shrine_glass")
            else
                set(x, 1, z, "wallygoo2:brick")
            end
        end
    end

    -- Layer 2: Glow accents
    for _, corner in ipairs({{-2, -2}, {2, -2}, {-2, 2}, {2, 2}}) do
        set(corner[1], 2, corner[2], "wallygoo2:shrine_glow")
    end

    -- Layer 3: Roof
    for x = -2, 2 do
        for z = -2, 2 do
            set(x, 3, z, "wallygoo2:brick")
        end
    end

    -- Particle effect
    minetest.add_particlespawner({
        amount = 20,
        time = 0.5,
        minpos = vector.add(pos, {x = -0.5, y = 0, z = -0.5}),
        maxpos = vector.add(pos, {x = 0.5, y = 3, z = 0.5}),
        minvel = {x = 0, y = 1, z = 0},
        maxvel = {x = 0, y = 2, z = 0},
        texture = "wallgoo_particle.png",
    })

    -- Sound effect
    minetest.sound_play("wallgoo_shrine_spawn", {pos = pos, gain = 1.0, max_hear_distance = 16})

    minetest.log("action", "[Wallgoo2] Shrine spawned at " .. minetest.pos_to_string(pos))
end

--========================
-- 🧱 Trapdoor System (Closed + 4 Open Directions)
--========================

-- Trapdoor nodeboxes by facedir
local trapdoor_open_boxes = {
    [0] = {{0.375, -0.5, -0.5, 0.5, 0.5, 0.5}},   -- +X
    [1] = {{-0.5, -0.5, 0.375, 0.5, 0.5, 0.5}},   -- +Z
    [2] = {{-0.5, -0.5, -0.5, -0.375, 0.5, 0.5}}, -- -X
    [3] = {{-0.5, -0.5, -0.5, 0.5, 0.5, -0.375}}, -- -Z
}

-- Closed trapdoor
minetest.register_node("wallygoo2:trapdoor", {
    description = "Wallgoo Trapdoor (Closed)",
    drawtype = "nodebox",
    tiles = {"wallgoo_trapdoor.png"},
    paramtype = "light",
    paramtype2 = "facedir",
    groups = {choppy = 2},
    sounds = default.node_sound_wood_defaults(),
    drop = "wallygoo2:trapdoor",
    node_box = {
        type = "fixed",
        fixed = {
            {-0.5, -0.5, -0.5, 0.5, -0.375, 0.5},
        },
    },
    on_rightclick = function(pos, node)
        local open_name = "wallygoo2:trapdoor_open_" .. tostring(node.param2 or 0)
        if minetest.registered_nodes[open_name] then
            minetest.swap_node(pos, {name = open_name, param2 = node.param2})
        else
            minetest.log("error", "[Wallgoo2] Invalid trapdoor param2: " .. tostring(node.param2))
        end
    end,
    on_place = function(itemstack, placer, pointed_thing)
        local dir = minetest.dir_to_facedir(placer:get_look_dir())
        local pos = pointed_thing.above
        minetest.set_node(pos, {name = "wallygoo2:trapdoor", param2 = dir})
        return itemstack
    end,
})

-- Open trapdoors for each direction
for i = 0, 3 do
    minetest.register_node("wallygoo2:trapdoor_open_" .. i, {
        description = "Wallgoo Trapdoor (Open)",
        drawtype = "nodebox",
        tiles = {"wallgoo_trapdoor.png"},
        paramtype = "light",
        paramtype2 = "facedir",
        groups = {choppy = 2, not_in_creative_inventory = 1},
        sounds = default.node_sound_wood_defaults(),
        drop = "wallygoo2:trapdoor",
        node_box = {
            type = "fixed",
            fixed = trapdoor_open_boxes[i],
        },
        on_rightclick = function(pos, node)
            minetest.swap_node(pos, {
                name = "wallygoo2:trapdoor",
                param2 = node.param2,
            })
        end,
    })
end

--========================
-- 🛠️ Shrine Spawner Crafting
--========================

minetest.register_craft({
    output = "wallygoo2:shrine_spawner",
    recipe = {
        {"wallygoo2:shrine_glow", "wallygoo2:shrine", "wallygoo2:shrine_glow"},
        {"wallygoo2:brick", "default:mese_crystal", "wallygoo2:brick"},
        {"wallygoo2:brick", "wallygoo2:brick", "wallygoo2:brick"},
    }
})
--========================
-- 🚪 Wallgoo Trapdoor System
--========================

-- Closed trapdoor
minetest.register_node("wallygoo2:trapdoor", {
    description = "Wallgoo Trapdoor (Closed)",
    drawtype = "nodebox",
    tiles = {"wallgoo_trapdoor.png"},
    paramtype = "light",
    paramtype2 = "facedir",
    groups = {choppy = 2},
    sounds = default.node_sound_wood_defaults(),
    drop = "wallygoo2:trapdoor",
    node_box = {
        type = "fixed",
        fixed = {
            {-0.5, -0.5, -0.5, 0.5, -0.375, 0.5}, -- horizontal slab
        },
    },
    on_rightclick = function(pos, node)
        local open_name = "wallygoo2:trapdoor_open_" .. tostring(node.param2 or 0)
        if minetest.registered_nodes[open_name] then
            minetest.swap_node(pos, {name = open_name, param2 = node.param2})
        else
            minetest.log("error", "[Wallgoo2] Invalid trapdoor param2: " .. tostring(node.param2))
        end
    end,
    on_place = function(itemstack, placer, pointed_thing)
        local dir = minetest.dir_to_facedir(placer:get_look_dir())
        local pos = pointed_thing.above
        minetest.set_node(pos, {name = "wallygoo2:trapdoor", param2 = dir})
        return itemstack
    end,
})

-- Directional open trapdoors
local trapdoor_open_boxes = {
    [0] = {{0.375, -0.5, -0.5, 0.5, 0.5, 0.5}},   -- Facing +X
    [1] = {{-0.5, -0.5, 0.375, 0.5, 0.5, 0.5}},   -- Facing +Z
    [2] = {{-0.5, -0.5, -0.5, -0.375, 0.5, 0.5}}, -- Facing -X
    [3] = {{-0.5, -0.5, -0.5, 0.5, 0.5, -0.375}}, -- Facing -Z
}

for i = 0, 3 do
    minetest.register_node("wallygoo2:trapdoor_open_" .. i, {
        description = "Wallgoo Trapdoor (Open)",
        drawtype = "nodebox",
        tiles = {"wallgoo_trapdoor.png"},
        paramtype = "light",
        paramtype2 = "facedir",
        groups = {choppy = 2, not_in_creative_inventory = 1},
        sounds = default.node_sound_wood_defaults(),
        drop = "wallygoo2:trapdoor",
        node_box = {
            type = "fixed",
            fixed = trapdoor_open_boxes[i],
        },
        on_rightclick = function(pos, node)
            minetest.swap_node(pos, {
                name = "wallygoo2:trapdoor",
                param2 = node.param2,
            })
        end,
    })
end

