-- Template mapping
local template_map = {
    ["wallygoo2:portal_frame"] = "wallygoo2:portal_research_template",
    ["wallygoo2:shrine"] = "wallygoo2:shrine_research_template",
}

-- Create template stack
local function create_template(item_name)
    if not item_name or item_name == "" then return nil end

    local template_id = template_map[item_name]
    local template
    if template_id then
        template = ItemStack(template_id)
    else
        template = ItemStack("wallygoo2:research_template")
        local meta = template:get_meta()
        meta:set_string("item_name", item_name)
        meta:set_string("id", item_name:gsub(":", "_"))
        meta:set_string("description", "Research Template: " .. item_name)
    end

    return template
end

-- Register template items
minetest.register_craftitem("wallygoo2:research_template", {
    description = "Wallgoo Research Template",
    inventory_image = "wallgoo_template.png",
    stack_max = 1,
})

minetest.register_craftitem("wallygoo2:portal_research_template", {
    description = "Portal Research Template",
    inventory_image = "wallgoo_portal_research_template.png",
})

minetest.register_craftitem("wallygoo2:shrine_research_template", {
    description = "Shrine Research Template",
    inventory_image = "wallgoo_shrine_research_template.png",
})

-- Register research table node
minetest.register_node("wallygoo2:research_table", {
    description = "Wallgoo Research Table",
    tiles = {"wallgoo_research_top.png", "wallgoo_steel.png", "wallgoo_steel.png"},
    groups = {cracky = 1},
    sounds = default.node_sound_metal_defaults(),

    on_construct = function(pos)
        local meta = minetest.get_meta(pos)
        local inv = meta:get_inventory()
        inv:set_size("input", 1)
        inv:set_size("output", 1)
        meta:set_string("infotext", "Wallgoo Research Table")
    end,

    on_rightclick = function(pos, node, clicker)
        local pname = clicker:get_player_name()
        minetest.show_formspec(pname, "wallygoo2:research_table_" .. minetest.pos_to_string(pos), table.concat({
            "formspec_version[4]",
            "size[10,9]",
            "label[0.5,0.5;Insert an item to research]",
            "list[nodemeta:", pos.x, ",", pos.y, ",", pos.z, ";input;3,1;1,1;]",
            "label[4.2,1.2;→]",
            "list[nodemeta:", pos.x, ",", pos.y, ",", pos.z, ";output;5,1;1,1;]",
            "button[6.5,1;2,1;confirm;Research]",
            "list[current_player;main;1,4.5;8,4;]",
            "listring[nodemeta:", pos.x, ",", pos.y, ",", pos.z, ";input]",
            "listring[nodemeta:", pos.x, ",", pos.y, ",", pos.z, ";output]",
            "listring[current_player;main]",
        }))
    end,
})

-- Handle form submission
minetest.register_on_player_receive_fields(function(player, formname, fields)
    if not player or not player:is_player() then return end
    if not formname or not fields or not fields.confirm then return end
    if not formname:find("wallygoo2:research_table_") then return end

    local pname = player:get_player_name()
    local pos_str = formname:match("wallygoo2:research_table_(.+)")
    local pos = pos_str and minetest.string_to_pos(pos_str)
    if not pos then
        minetest.chat_send_player(pname, "Error: Could not parse position.")
        return
    end

    local meta = minetest.get_meta(pos)
    local inv = meta and meta:get_inventory()
    if not inv then
        minetest.chat_send_player(pname, "Error: No inventory found.")
        return
    end

    local stack = inv:get_stack("input", 1)
    if not stack or stack:is_empty() then
        minetest.chat_send_player(pname, "Insert an item first!")
        return
    end

    local item_name = stack:get_name()
    if item_name == "" then
        minetest.chat_send_player(pname, "Error: Item has no name.")
        return
    end

    stack:take_item()
    inv:set_stack("input", 1, stack)

    local template = create_template(item_name)
    if not template or template:is_empty() then
        minetest.chat_send_player(pname, "Error: Could not create template item.")
        return
    end

    if inv:get_stack("output", 1):is_empty() then
        inv:set_stack("output", 1, template)
        minetest.chat_send_player(pname, "You researched: " .. item_name)
        minetest.sound_play("wallygoo2_research", {pos = pos, gain = 1.0})
    else
        minetest.chat_send_player(pname, "Output slot is full! Remove the previous template first.")
    end
end)

-- Crafting recipe
minetest.register_craft({
    output = "wallygoo2:research_table",
    recipe = {
        {"default:copper_ingot", "default:mese_crystal", "default:copper_ingot"},
        {"wallygoo2:brick", "wallygoo2:brick", "wallygoo2:brick"},
    }
})

