-- util.lua
-- Utility helpers for the musket mod

local util = {}

--------------------------------------------------------------------------------
-- Time
--------------------------------------------------------------------------------

--- Returns the current game time in seconds.
function util.get_time()
	return minetest.get_gametime()
end

--------------------------------------------------------------------------------
-- Audio
--------------------------------------------------------------------------------

--- Plays a sound at a world position.
-- @param pos  table  {x, y, z} world position
-- @param sound string|table  sound name or sound parameter table
-- @param gain  number  optional volume override (default 1.0)
-- @param max_hear_distance  number  optional hearing radius (default 64)
function util.play_sound(pos, sound, gain, max_hear_distance)
	if not sound then return end
	minetest.sound_play(sound, {
		pos              = pos,
		gain             = gain or 1.0,
		max_hear_distance = max_hear_distance or 64,
	})
end

--------------------------------------------------------------------------------
-- Particles
--------------------------------------------------------------------------------

--- Spawns a burst of smoke particles at a position.
-- @param pos       table   {x, y, z} origin
-- @param dir       table   {x, y, z} direction vector (need not be normalised)
-- @param amount    integer number of particles to emit
-- @param spread    number  random velocity spread radius
-- @param lifetime  number  particle lifetime in seconds
function util.spawn_smoke(pos, dir, amount, spread, lifetime)
	-- Base velocity: half the direction plus a slight upward drift
	local bx = dir.x * 0.5
	local by = dir.y * 0.5 + 0.2
	local bz = dir.z * 0.5

	local half = spread * 0.5

	for _ = 1, amount do
		minetest.add_particle({
			pos      = vector.new(pos.x, pos.y, pos.z),
			velocity = {
				x = bx + math.random() * spread - half,
				y = by + math.random() * spread - half,
				z = bz + math.random() * spread - half,
			},
			acceleration        = {x = 0, y = 0.05, z = 0},
			expirationtime      = lifetime + math.random() * 0.3, -- slight lifetime variance
			size                = math.random(6, 14),
			collisiondetection  = false,
			vertical            = false,
			texture             = "smoke_particle.png^[opacity:160",
		})
	end
end

--- Spawns a single muzzle-flash particle at a position.
-- @param pos  table  {x, y, z} muzzle world position
-- @param dir  table  {x, y, z} barrel direction vector
function util.spawn_muzzle_flash(pos, dir)
	minetest.add_particle({
		pos      = vector.add(pos, vector.multiply(dir, 0.3)),
		velocity = vector.multiply(dir, 1.5),
		acceleration        = {x = 0, y = -0.5, z = 0},
		expirationtime      = 0.1,
		size                = math.random(10, 16),
		collisiondetection  = false,
		vertical            = false,
		texture             = "muzzle_flash.png",
		glow                = 14,
	})
end

--------------------------------------------------------------------------------
-- Math / Geometry
--------------------------------------------------------------------------------

--- Clamps a value between min and max.
function util.clamp(v, min, max)
	return math.max(min, math.min(max, v))
end

--- Returns a random float in [lo, hi].
function util.rand_range(lo, hi)
	return lo + math.random() * (hi - lo)
end

--- Applies angular spread to a direction vector.
-- Useful for musket inaccuracy simulation.
-- @param dir    table  normalised {x, y, z} direction
-- @param angle  number max spread angle in radians
-- @return table  perturbed direction (not renormalised — call vector.normalize if needed)
function util.spread_direction(dir, angle)
	if angle <= 0 then return dir end
	local yaw   = math.atan2(dir.x, dir.z) + util.rand_range(-angle, angle)
	local pitch = math.asin(util.clamp(dir.y, -1, 1)) + util.rand_range(-angle, angle)
	local cp    = math.cos(pitch)
	return {
		x = math.sin(yaw) * cp,
		y = math.sin(pitch),
		z = math.cos(yaw) * cp,
	}
end

--------------------------------------------------------------------------------
-- Entity helpers
--------------------------------------------------------------------------------

--- Returns the eye-level position of a player or entity.
-- @param obj  ObjectRef
-- @return table  {x, y, z}
function util.get_eye_pos(obj)
	local pos = obj:get_pos()
	if obj:is_player() then
		local eye_height = obj:get_properties().eye_height or 1.625
		pos.y = pos.y + eye_height
	end
	return pos
end

--- Deals damage to an entity, applying a brief knockback impulse.
-- @param target  ObjectRef  victim
-- @param damage  number     hit points
-- @param dir     table      normalised {x, y, z} impulse direction
-- @param force   number     knockback magnitude (default 4)
function util.apply_damage(target, damage, dir, force)
	if not target or not target:get_pos() then return end
	target:punch(target, 1.0, {
		full_punch_interval = 1.0,
		damage_groups       = {fleshy = damage},
	}, dir)
	if force and force > 0 and dir then
		local vel = target:get_velocity()
		if vel then
			target:set_velocity({
				x = vel.x + dir.x * force,
				y = vel.y + dir.y * force + 2,
				z = vel.z + dir.z * force,
			})
		end
	end
end

--------------------------------------------------------------------------------
-- Cooldown / rate-limit helpers
--------------------------------------------------------------------------------

-- Internal table: player_name -> last_fire_time
local _cooldowns = {}

--- Records the current time as the last action time for a player.
function util.set_cooldown(player_name)
	_cooldowns[player_name] = minetest.get_gametime()
end

--- Returns true if enough time has elapsed since the last recorded action.
-- @param player_name string
-- @param interval    number  required gap in seconds
function util.cooldown_elapsed(player_name, interval)
	local last = _cooldowns[player_name]
	if not last then return true end
	return (minetest.get_gametime() - last) >= interval
end

--- Cleans up cooldown state when a player leaves.
minetest.register_on_leaveplayer(function(player)
	_cooldowns[player:get_player_name()] = nil
end)

--------------------------------------------------------------------------------

return util
