-- weapons.lua
local modpath = minetest.get_modpath("muskets")
local util    = dofile(modpath .. "/util.lua")
local sounds  = dofile(modpath .. "/sounds.lua")

local weapons = {
    jezail = {
        description  = "Jezail Musket",
        texture      = "jezail.png",
        damage       = 57,
        velocity     = 80,
        gravity      = 3.0,
        spread       = 3.0,
        reload_time  = 8.5,
        smoke_amount = 18,
        ads_mult     = 0.25,
        move_penalty = 0.45,
    },
    whitworth = {
        description  = "Whitworth Rifle",
        texture      = "whitworth.png",
        damage       = 50,
        velocity     = 110,
        gravity      = 1.5,
        spread       = 2.0,
        reload_time  = 9.0,
        smoke_amount = 22,
        ads_mult     = 0.18,
        move_penalty = 0.35,
    },
    springfield_m1861 = {
        description  = "Springfield M1861",
        texture      = "springfield_m1861.png",
        damage       = 32,
        velocity     = 75,
        gravity      = 4.0,
        spread       = 4.0,
        reload_time  = 7.5,
        smoke_amount = 16,
        ads_mult     = 0.30,
        move_penalty = 0.50,
    },
    pennsylvanian = {
        description  = "Pennsylvanian Rifle",
        texture      = "pennsylvanian.png",
        damage       = 70,
        velocity     = 80,
        gravity      = 5.5,
        spread       = 2.0,
        reload_time  = 12.5,
        smoke_amount = 12,
        ads_mult     = 0.35,
        move_penalty = 0.55,
    },
    tower_contract = {
        description  = "Tower Contract Rifle",
        texture      = "tower_contract.png",
        damage       = 86,
        velocity     = 120,
        gravity      = 1.0,
        spread       = 3.0,
        reload_time  = 12.0,
        smoke_amount = 28,
        ads_mult     = 0.12,
        move_penalty = 0.25,
    },
}

local player_state = {}

local function get_state(player)
    local name = player:get_player_name()
    if not player_state[name] then
        player_state[name] = {
            ads            = false,
            reloading      = false,
            last_shot      = 0,
            reload_penalty = 1.0,
            wield_item     = nil,
        }
    end
    return player_state[name]
end

minetest.register_on_leaveplayer(function(player)
    player_state[player:get_player_name()] = nil
end)

local function calc_spread_rad(w, user, state)
    local spread = w.spread
    local vel = user:get_velocity()
    if math.abs(vel.x) + math.abs(vel.z) > 0.1 then
        spread = spread * 1.6
    end
    if user:get_player_control().sneak then
        spread = spread * 0.6
    end
    if state.ads then
        spread = spread * w.ads_mult
    end
    return math.rad(spread)
end

local function notify(player, msg)
    minetest.chat_send_player(player:get_player_name(), msg)
end

local RAY_STEP_MAX        = 6.0
local PROJECTILE_LIFETIME = 8

minetest.register_entity("muskets:projectile", {
    initial_properties = {
        physical     = false,
        visual       = "sprite",
        visual_size  = {x = 0.15, y = 0.15},
        textures     = {"musket_shot.png"},
        collisionbox = {0, 0, 0, 0, 0, 0},
        is_visible   = true,
        pointable    = false,
    },
    _damage  = 10,
    _dir     = {x = 0, y = 0, z = 1},
    _shooter = nil,
    _timer   = 0,
    on_step = function(self, dtime)
        self._timer = self._timer + dtime
        if self._timer > PROJECTILE_LIFETIME then
            self.object:remove()
            return
        end
        local pos = self.object:get_pos()
        if not pos then return end
        local vel      = self.object:get_velocity()
        local travel   = vector.multiply(vel, dtime)
        local ray_len  = vector.length(travel)
        local ray_dir  = ray_len > 0 and vector.divide(travel, ray_len) or self._dir
        local cast_len = math.min(ray_len, RAY_STEP_MAX)
        local ray_end  = vector.add(pos, vector.multiply(ray_dir, cast_len))
        local ray = minetest.raycast(pos, ray_end, true, true)
        for hit in ray do
            if hit.type == "object" then
                local obj = hit.ref
                if obj == self._shooter then break end
                local ent = obj:get_luaentity()
                if ent and ent.name == "muskets:projectile" then break end
                obj:punch(self.object, 1.0, {
                    full_punch_interval = 1.0,
                    damage_groups       = {fleshy = self._damage},
                }, self._dir)
                local kvel = obj:get_velocity()
                if kvel then
                    obj:set_velocity({
                        x = kvel.x + self._dir.x * 3,
                        y = kvel.y + 2,
                        z = kvel.z + self._dir.z * 3,
                    })
                end
                util.spawn_smoke(hit.intersection_point, self._dir, 6, 0.6, 1.2)
                self.object:remove()
                return
            elseif hit.type == "node" then
                util.spawn_smoke(hit.intersection_point, self._dir, 8, 0.8, 1.6)
                self.object:remove()
                return
            end
        end
    end,
})

for id, w in pairs(weapons) do
    local itemname = "muskets:" .. id

    local function on_use(itemstack, user, _pointed_thing)
        if not user:is_player() then return itemstack end
        local state = get_state(user)
        local name  = user:get_player_name()

        if state.reloading then
            notify(user, "Still reloading...")
            return itemstack
        end

        local now = util.get_time()
        if now - state.last_shot < w.reload_time then
            notify(user, "Weapon not ready.")
            return itemstack
        end

        local spread_rad = calc_spread_rad(w, user, state)
        local dir        = util.spread_direction(user:get_look_dir(), spread_rad)
        local eye_pos    = util.get_eye_pos(user)
        local spawn_pos  = vector.add(eye_pos, vector.multiply(dir, 0.4))

        local obj = minetest.add_entity(spawn_pos, "muskets:projectile")
        if obj then
            local ent    = obj:get_luaentity()
            ent._damage  = w.damage
            ent._dir     = dir
            ent._shooter = user
            obj:set_velocity(vector.multiply(dir, w.velocity))
            obj:set_acceleration({x = 0, y = -w.gravity, z = 0})
        end

        util.play_sound(eye_pos, sounds.fire)
        util.spawn_smoke(vector.add(eye_pos, vector.multiply(dir, 0.5)), dir, w.smoke_amount, 1.2, 2.0)
        util.spawn_muzzle_flash(eye_pos, dir)
        user:add_velocity(vector.multiply(dir, -0.6))

        state.last_shot      = now
        state.reloading      = true
        state.reload_penalty = w.move_penalty
        state.wield_item     = user:get_wielded_item()

        if not state.ads then
            user:set_physics_override({speed = w.move_penalty})
        end

        minetest.after(w.reload_time, function()
            local p = minetest.get_player_by_name(name)
            if not p then return end
            local st = get_state(p)
            st.reloading = false
            if not st.ads then
                p:set_physics_override({speed = 1.0})
            end
            notify(p, w.description .. " reloaded.")
            util.play_sound(p:get_pos(), sounds.reload)
        end)

        return itemstack
    end

    local function on_secondary_use(itemstack, user, _pointed_thing)
        if not user:is_player() then return itemstack end
        local state = get_state(user)
        state.ads = not state.ads

        if state.ads then
            notify(user, "Aiming down sights")
            local phys = user:get_physics_override()
            user:set_physics_override({speed = (phys.speed or 1.0) * w.move_penalty})
            user:set_fov(0.6, true, 0.2)
        else
            notify(user, "Hip fire")
            user:set_physics_override({speed = 1.0})
            user:set_fov(1.0, true, 0.2)
        end

        return itemstack
    end

    minetest.register_tool(itemname, {
        description      = w.description,
        inventory_image  = w.texture,
        stack_max        = 1,
        on_use           = on_use,
        on_secondary_use = on_secondary_use,
    })
end

minetest.register_globalstep(function(_dtime)
    for _, player in ipairs(minetest.get_connected_players()) do
        if not player or not player:is_player() then goto continue end
        local state = player_state[player:get_player_name()]
        if not state then goto continue end

        if state.reloading then
            local musket_name = state.wield_item and state.wield_item:get_name()
            if musket_name then
                local cur = player:get_wielded_item()
                if cur:get_name() ~= musket_name then
                    local inv       = player:get_inventory()
                    local list      = inv:get_list("main")
                    local wield_idx = player:get_wield_index()
                    for i, stack in ipairs(list) do
                        if stack:get_name() == musket_name then
                            inv:set_stack("main", wield_idx, stack)
                            inv:set_stack("main", i, cur)
                            break
                        end
                    end
                end
            end
            if not state.ads then
                local phys = player:get_physics_override()
                local cur  = phys and phys.speed or 1.0
                local pen  = state.reload_penalty or 1.0
                if cur > pen then
                    player:set_physics_override({speed = pen})
                end
            end
        elseif not state.ads then
            local phys = player:get_physics_override()
            if phys and phys.speed and phys.speed < 1.0 then
                player:set_physics_override({speed = 1.0})
            end
        end

        ::continue::
    end
end)
