local path = ""
if core then
    path = core.get_modpath("priority_queue")
else
    path = "."
end

local Class, Extend = dofile(path .. "/classes.lua")

local floor = math.floor
local MinHeap = Class()

local function _up_heap(self, idx)
    local values = self._values
    local priorities = self._priorities
    local keys = self._keys

    while idx > 1 do
        local parent_idx = floor(idx / 2)

        local vp = values[parent_idx]
        local vi = values[idx]
        local pp = priorities[parent_idx]
        local pi = priorities[idx]

        if pp <= pi then
            break
        end

        keys[vi], keys[vp] = parent_idx, idx

        values[idx], values[parent_idx], priorities[idx], priorities[parent_idx] = vp, vi, pp, pi

        idx = parent_idx
    end
end

local function _down_heap(self, idx)
    local s = self
    local values = s._values
    local priorities = s._priorities
    local keys = s._keys

    while true do
        local left_child_idx = 2 * idx
        local right_child_idx = 2 * idx + 1
        local smallest = idx
        local pi = priorities[idx]
        local ps = pi

        if left_child_idx <= s._size and priorities[left_child_idx] < priorities[smallest] then
            smallest = left_child_idx
        end

        if smallest == left_child_idx then
            ps = priorities[smallest]
        end

        if right_child_idx <= s._size and priorities[right_child_idx] < ps then
            smallest = right_child_idx
        end

        if smallest == idx then break end

        local vs = values[smallest]
        local vi = values[idx]

        if smallest == right_child_idx then
            ps = priorities[smallest]
        end

        keys[vi], keys[vs] = smallest, idx

        values[idx], values[smallest], priorities[idx], priorities[smallest] = vs, vi, ps, pi

        idx = smallest
    end
end

function MinHeap:_init()
    local s = self
    s._values = {}
    s._priorities = {}
    s._keys = {}
    s._size = 0
end

function MinHeap:insert(value, priority)
    local s = self
    local values = s._values
    local priorities = s._priorities
    local keys = s._keys

    local idx = keys[value]

    if idx == nil then
        s._size = s._size + 1
        values[s._size] = value
        priorities[s._size] = priority
        keys[value] = s._size
        _up_heap(s, s._size)
        return true
    else
        local old_priority = priorities[idx]

        if old_priority > priority then
            priorities[idx] = priority
            _up_heap(s, idx)
        elseif old_priority < priority then
            priorities[idx] = priority
            _down_heap(s, idx)
        end
        return false
    end
end

function MinHeap:extract()
    local s = self
    local values = s._values
    local priorities = s._priorities
    local keys = s._keys

    if s._size == 0 then
        return nil
    end

    if s._size == 1 then
        local value = values[1]
        local priority = priorities[1]
        values[1] = nil
        priorities[1] = nil
        s._size = 0
        keys[value] = nil
        return value, priority
    end

    local value = values[1]
    local priority = priorities[1]

    keys[value] = nil
    values[1] = values[s._size]
    values[s._size] = nil
    priorities[1] = priorities[s._size]
    priorities[s._size] = nil
    s._size = s._size - 1
    keys[values[1]] = 1
    _down_heap(s, 1)
    return value, priority
end

function MinHeap:peek()
    local value = self._values[1]
    if value == nil then
        return nil
    end
    local priority = self._priorities[1]
    return value, priority
end

function MinHeap:search(value)
    if self._keys[value] == nil then
        return false
    end
    return true
end

function MinHeap:delete(value)
    local s = self
    local values = s._values
    local priorities = s._priorities
    local keys = s._keys

    local idx = keys[value]

    if idx == nil then
        return false
    end

    keys[value] = nil

    if idx == s._size then
        values[s._size] = nil
        priorities[s._size] = nil
        s._size = s._size - 1
        return true
    end

    local old_priority = priorities[idx]

    values[idx] = values[s._size]
    values[s._size] = nil
    local priority = priorities[s._size]
    priorities[idx] = priority
    priorities[s._size] = nil
    s._size = s._size - 1

    keys[values[idx]] = idx

    if old_priority > priority then
        _down_heap(s, idx)
    end

    if old_priority < priority then
        _up_heap(s, idx)
    end

    return true
end

function MinHeap:clear()
    self._values = {}
    self._priorities = {}
    self._keys = {}
    self._size = 0
end

--------------------------------------------------------------------------------

local MaxHeap = Extend(MinHeap)

function MaxHeap:_init()
    self:_super()
end

function MaxHeap:insert(value, priority)
    MinHeap.insert(self, value, -priority)
end

function MaxHeap:extract(value, priority)
    local value, priority = MinHeap.extract(self)
    return value, -priority
end

function MaxHeap:peek(value, priority)
    local value, priority = MinHeap.peek(self)
    return value, -priority
end

return MinHeap, MaxHeap
