--- Utilities for hypertext used in Minetest formspecs.
-- @module advtrains_doc_integration.hypertext
-- @alias H
local H = {}
local S = minetest.get_translator("advtrains_doc_integration")
local utils = advtrains_doc_integration.utils

--- Create a plain text hypertext element.
-- @tparam string str The content of the element.
-- @tparam[opt] boolean noescape Whether to avoid escaping the content.
-- @treturn string The hypertext element containing the text.
function H.plain(str, noescape)
	str = tostring(str)
	if noescape then
		return str
	end
	return (string.gsub(str, "([<>])", [[\%1]]))
end

local function addtag(tag, attrs, str, noescape)
	local attrlist = {}
	for k, v in pairs(attrs) do
		table.insert(attrlist, (" %s=%s"):format(k, v))
	end
	return ("<%s%s>%s</%s>"):format(tag, table.concat(attrlist), H.plain(str, noescape), tag)
end

--- Create a hypertext element with monospace text.
-- @tparam string str The content of the element.
-- @tparam[opt] boolean noescape Whether to avoid escaping the content.
-- @treturn string The hypertext element containing the text.
function H.mono(str, noescape)
	return addtag("mono", {}, str, noescape)
end

--- Create a hypertext element with bold text.
-- @tparam string str The content of the element.
-- @tparam[opt] boolean noescape Whether to avoid escaping the content.
-- @treturn string The hypertext element containing the text.
function H.bold(str, noescape)
	return addtag("b", {}, str, noescape)
end

--- Create a hypertext element with italic text.
-- @tparam string str The content of the element.
-- @tparam[opt] boolean noescape Whether to avoid escaping the content.
-- @treturn string The hypertext element containing the text.
function H.italic(str, noescape)
	return addtag("i", {}, str, noescape)
end

--- Create a hypertext element with text formatted as header.
-- @tparam string str The content of the element.
-- @tparam[opt] boolean noescape Whether to avoid escaping the content.
-- @treturn string The hypertext element containing the text.
function H.header(str, noescape)
	return H.bold(str, noescape)
end

--- Create a hypertext element with text formatted as an item in a list.
-- @tparam string str The content of the element.
-- @tparam[opt] boolean noescape Whether to avoid escaping the content.
-- @treturn string The hypertext element containing the text.
function H.listitem(str, noescape)
	return "• " .. H.plain(str, noescape)
end

--- Create a hypertext element with an action.
-- @tparam string action The name of the action.
-- @tparam string str The content of the element.
-- @tparam[opt] boolean noescape Whether to avoid escaping the content.
-- @treturn string The hypertext element containing the text.
function H.action(action, str, noescape)
	return addtag("action", {name = action},
		addtag("style", {color = "cyan"}, str, noescape), true)
end

--- Describe a soundspec.
-- @tparam[opt] string action The name of the action.
-- @tparam SimpleSoundSpec soundspec The soundspec to describe.
-- @treturn string The hypertext element describing the soundspec.
function H.describe_sound(action, soundspec)
	if not soundspec then
		return S("Undefined")
	end
	local sounddesc = H.mono(soundspec.name)
	if action then
		sounddesc = H.action(action, sounddesc, true)
	end
	return sounddesc
end

--- Describe a coupler type.
-- @tparam string coupler The name of the coupler.
-- @treturn string The hypertext element describing the coupler.
function H.describe_coupler(coupler)
	local name = utils.get_coupler_name(coupler)
	if name then
		return ("%s (%s)"):format(H.plain(name), H.mono(coupler))
	else
		return H.mono(coupler)
	end
end

--- Describe an item.
-- @tparam ItemStack item The item to describe
-- @treturn string The hypertext element describing the item.
function H.describe_item(item)
	item = ItemStack(item)
	if item:is_empty() then
		return S("Emptyness")
	end
	local shortdesc = item:get_short_description()
	local name = item:get_name()
	local count = item:get_count()
	return ("%s (%s): %d"):format(H.plain(shortdesc), H.mono(name), count)
end

--- Describe a function.
-- @tparam function f The function to describe.
-- @treturn string|nil The hypertext element describing the function if a sufficiently informative description is available.
function H.describe_function(f)
	local bctype, bcdesc = advtrains_doc_integration.bc.tostring(f)
	if bctype and bcdesc then
		return S("Compiled function (@1):", H.mono(bctype)) .. "\n" .. H.mono(bcdesc)
	end
	local dinfo = debug.getinfo(f)
	if not dinfo then
		return nil
	elseif dinfo.isvararg then
		return S("Variadic function")
	elseif dinfo.nparams then
		return S("@1-ary function", ("%d"):format(dinfo.nparams))
	end
	return nil
end

return H
