local S = minetest.get_translator "advtrains_doc_integration"
local NS = tostring
local D = advtrains_doc_integration.describe
local H = advtrains_doc_integration.hypertext
local bc = advtrains_doc_integration.bc
local utils = advtrains_doc_integration.utils
local fsescape = minetest.formspec_escape

local function S2(a, b)
	return S(a, S(b))
end

local function addlist(lst, tbl, title, fallback1, fallback2, mapf)
	if not tbl then
		if fallback2 then
			table.insert(lst, fallback2)
		elseif fallback2 == false and fallback1 then
			table.insert(lst, fallback1)
		end
	elseif next(tbl) ~= nil then
		table.insert(lst, title)
		for k, v in pairs(tbl) do
			if mapf then
				k = mapf(k, v)
			end
			table.insert(lst, H.listitem(k, true))
		end
	elseif fallback1 then
		table.insert(lst, fallback1)
	end
end

local function describe_function(f)
	if not f then
		return S("Undefined")
	end
	return H.describe_function(f) or S("Defined")
end

local function blankline(st)
	return table.insert(st, "")
end

local wlivprev = {}

local function get_livery_preview_selection(pname, itemname)
	return (wlivprev[pname] or {})[itemname] or "default"
end

local function get_livery_preview(itemname, id)
	local tx = (advtrains_doc_integration.prototypes[itemname] or {}).livery_textures
	return tx[id] or tx["default"]
end

local function render_livery_textures(pname, itemname)
	local str = table.concat(utils.map(get_livery_preview(itemname, get_livery_preview_selection(pname, itemname)), fsescape), ",")
	return str
end

local function set_livery_preview_selection(pname, itemname, id)
	local t = wlivprev[pname]
	if not t then
		t = {}
		wlivprev[pname] = t
	end
	t[itemname] = id
end

local docfsdim = {
	x0 = doc.FORMSPEC.ENTRY_START_X, y0 = doc.FORMSPEC.ENTRY_START_Y,
	x1 = doc.FORMSPEC.ENTRY_END_X+0.25, y1 = doc.FORMSPEC.ENTRY_END_Y+0.625,
}
docfsdim.w, docfsdim.h = docfsdim.x1-docfsdim.x0, docfsdim.y1-docfsdim.y0
docfsdim.nh = docfsdim.h/2
docfsdim.nw = docfsdim.nh
docfsdim.nx0, docfsdim.ny0 = docfsdim.x0+0.275, docfsdim.y0
docfsdim.cx0, docfsdim.cy0 = docfsdim.x0+docfsdim.nw, docfsdim.y0
docfsdim.cw, docfsdim.ch = docfsdim.w-docfsdim.nw, docfsdim.h+0.875
docfsdim.lx0, docfsdim.ly0 = docfsdim.x0, docfsdim.y0+docfsdim.nh
docfsdim.lw, docfsdim.lh = docfsdim.nw, docfsdim.nh

local function renderer_from_hypertext(f)
	return function(prototype, pname)
		local ht = f(prototype, pname)
		if type(ht) == "table" then
			ht = table.concat(ht, "\n")
		end
		if not ht or ht == "" then
			return
		end
		local fstext = {
			string.format("hypertext[%f,%f;%f,%f;entry_body;%s]", docfsdim.cx0, docfsdim.cy0, docfsdim.cw, docfsdim.ch, fsescape(ht)),
		}
		if prototype.mesh then
			local mesh = fsescape(prototype.mesh)
			local textures = render_livery_textures(pname, prototype.name)
			table.insert(fstext, string.format("model[%f,%f;%f,%f;%s;%s;%s;%f,%f]",
				docfsdim.lx0, docfsdim.ly0, docfsdim.lw, docfsdim.lh, "wagon_model", mesh, textures, -30, 135))
		end
		if get_livery_preview_selection(pname, prototype.name) ~= "default" and not prototype.dlxtrains_livery then
			table.insert(fstext, string.format("button[%f,%f;%f,1;wagon_reset_preview;%s]",
				docfsdim.lx0, doc.FORMSPEC.HEIGHT-0.5, docfsdim.lw,
				fsescape(S "Reset livery preview")))
		end
		return table.concat(fstext, "\n")
	end
end

local wtabopen = {}
local wtabs = {}

wtabs.summary = renderer_from_hypertext(function(prototype)
	local desctext = {}
	if prototype._doc_wagon_longdesc then
		table.insert(desctext, tostring(prototype._doc_wagon_longdesc))
		blankline(desctext)
	end
	table.insert(desctext, H.header(S("Basic Information")))
	table.insert(desctext, S("Itemstring: @1", H.mono(prototype.name)))
	addlist(desctext, prototype.drops, S("Drops:"), S("Drops nothing"), false, function(_, v) return H.describe_item(v) end)
	addlist(desctext, prototype.drives_on, S("Drives on:"), nil, nil, H.mono)
	addlist(desctext, prototype.coupler_types_front, S("Compatible front couplers:"), S2("Front coupler: @1", "Absent"), S2("Front coupler: @1", "Universal"), H.describe_coupler)
	addlist(desctext, prototype.coupler_types_back, S("Compatible rear couplers:"), S2("Rear coupler: @1", "Absent"), S2("Rear coupler: @1", "Universal"), H.describe_coupler)
	table.insert(desctext, S("Wagon span: @1", prototype.wagon_span and D.length(2*prototype.wagon_span) or S("Undefined")))
	table.insert(desctext, S("Maximum speed: @1", prototype.max_speed and D.speed(prototype.max_speed) or S("Undefined")))
	table.insert(desctext, S2("Motive power: @1", prototype.is_locomotive and "Present" or "Absent"))
	table.insert(desctext, S("Horn sound: @1", H.describe_sound("playhorn", prototype.horn_sound)))
	if prototype.doors.open.sound or prototype.doors.close.sound then
		table.insert(desctext, S("Door sound: @1 (when opening), @2 (when closing)",
			H.describe_sound("playdooropen", prototype.doors.open.sound),
			H.describe_sound("playdoorclose", prototype.doors.close.sound)))
	else
		table.insert(desctext, S2("Door sound: @1", "Undefined"))
	end

	blankline(desctext)
	table.insert(desctext, H.header(S("Wagon Capacity")))
	table.insert(desctext, S("Passenger seats: @1", prototype.max_passengers))
	table.insert(desctext, S("Driver seats: @1", prototype.max_drivers))
	table.insert(desctext, S("Cargo inventory: @1", prototype.has_inventory and S("Present") or S("Absent")))
	if techage and prototype.techage_liquid_capacity then
		table.insert(desctext, S("Liquid Capacity (Techage): @1", string.format("%d", prototype.techage_liquid_capacity)))
	end

	blankline(desctext)
	table.insert(desctext, H.header(S("Wagon Appearance")))
	table.insert(desctext, S("Mesh: @1", prototype.mesh and H.mono(prototype.mesh) or "None"))
	addlist(desctext, prototype.textures, S("Textures:"), S("No textures"), false, function(_, v) return H.mono(v) end)
	table.insert(desctext, S("Livery customization with bike painter: @1",
		prototype.set_livery and S "Supported" or S "Unsupported"))

	return desctext
end)

local function render_seat_capacity(desctext, header, tbl)
	if next(tbl) then
		table.insert(desctext, H.header(header))
		for k, v in utils.spairs(tbl) do
			table.insert(desctext, ("%s: %d"):format(v.name and H.escape(v.name) or H.mono(k), v.count))
		end
	end
end

wtabs.capacity = renderer_from_hypertext(function(prototype)
	local desctext = {}

	if next(prototype.seat_groups or {}) then
		local driver, pax = {}, {}
		for k, v in pairs(prototype.seat_groups) do
			if v.driving_ctrl_access then
				driver[k] = v
			else
				pax[k] = v
			end
		end
		render_seat_capacity(desctext, S "Driver area capacity:", driver)
		render_seat_capacity(desctext, S "Passenger area capacity:", pax)

		local attachment_offset_support = S("Unsupported")
		if advtrains_attachment_offset_patch then
			local t = advtrains_attachment_offset_patch
			if prototype.get_on == t.get_on_override and prototype.get_off == t.get_off_override then
				attachment_offset_support = S("Supported")
			end
		end
		table.insert(desctext, S("Proper player attachment positioning: @1", attachment_offset_support))
	end
	if techage and prototype.techage_liquid_capacity then
		table.insert(desctext, S("Liquid Capacity (Techage): @1", string.format("%d", prototype.techage_liquid_capacity)))
	end
	if prototype.has_inventory then
		table.insert(desctext, H.header(S "Cargo inventory size:"))
		for k, v in utils.spairs(prototype.inventory_list_sizes) do
			table.insert(desctext, H.listitem(("%s: %d"):format(H.mono(k), v), true))
		end
	end

	return desctext
end)

wtabs.bytecode = renderer_from_hypertext(function(prototype, pname)
	local desctext = {}
	if core.is_singleplayer() or core.check_player_privs(pname, "server") then
		if bc.is_enabled() then
			table.insert(desctext, S("Warning: bytecode parsing is experimental."))
		else
			table.insert(desctext, S("Bytecode parsing is disabled. Only basic information will be provided."))
		end
		if not core.is_singleplayer() then
			table.insert(desctext, S("Note that it is recommended to keep bytecode parsing disabled on multiplayer servers."))
		end
	end

	for k, v in pairs {
		custom_on_activate = NS("Custom instantiation callback: @1"),
		custom_on_step = NS("Custom step function: @1"),
		custom_on_velocity_change = NS("Custom velocity change callback: @1"),
	} do
		table.insert(desctext, S(v, describe_function(prototype[k])))
	end

	return desctext
end)

wtabs.liv_dlx = renderer_from_hypertext(function(prototype, pname)
	local dlxlivdef = prototype.dlxtrains_livery
	if not dlxlivdef then
		return
	end
	local desctext = {}
	table.insert(desctext, H.header(S "Livery presets:"))
	for i = 0, dlxlivdef.count - 1 do
		local baseid = "dlx:" .. i
		local livsel = get_livery_preview_selection(pname, prototype.name)
		if livsel == "default" then
			livsel = "dlx:0"
		end
		local lt = {}
		for k, v in ipairs {
			{"", S "Fresh livery"},
			{"w", S "Weathered livery"},
		} do
			local id = baseid .. v[1]
			local desc = H.escape(v[2])
			if livsel == id then
				desc = H.bold(desc, true)
			end
			lt[k] = H.action("preview:" .. id, desc, true)
		end
		local livname = H.mono(dlxlivdef[i].code)
		if dlxlivdef[i].name then
			livname = H.escape(dlxlivdef[i].name)
		end
		local desc = S("@1 (@2) by @3: @4",
			livname, ("%s, %s"):format(lt[1], lt[2]),
			H.escape(dlxlivdef[i].designer or S "Unknown author"),
			dlxlivdef[i].notes and H.escape(dlxlivdef[i].notes)
				or S("Standard preset for @1", H.mono(dlxlivdef[i].code)))
		table.insert(desctext, H.listitem(desc, true))
	end
	if dlxtrains.use_advtrains_livery_designer then
		table.insert(desctext, S "This wagon can also by customized using Marnack's Advtrains livery designer.")
	end
	return desctext
end)

wtabs.liv_mul = renderer_from_hypertext(function(prototype, pname)
	local bikelivdef = prototype.livery_definition
	if not bikelivdef then
		return
	end
	local desctext = {}
	table.insert(desctext, H.header(S "Livery components:"))
	for _, v in ipairs(bikelivdef.components) do
		table.insert(desctext, H.listitem(v.description))
	end
	local livsel = get_livery_preview_selection(pname, prototype.name)
	table.insert(desctext, H.header(S "Livery presets:"))
	for k, v in ipairs(bikelivdef.presets) do
		local name, id = H.escape(v.description), "mul:" .. k
		if livsel == id then
			name = H.bold(name, true)
		end
		table.insert(desctext, H.listitem(H.action("preview:" .. id, name, true), true))
	end
	return desctext
end)

wtabs.liv_alt = renderer_from_hypertext(function(prototype, pname)
	local atlivdef = prototype.advtrains_livery_tools
	if not atlivdef then
		return
	end
	local desctext = {}
	local livgroups = {}
	local livsel = get_livery_preview_selection(pname, prototype.name)
	for k, dname in ipairs(atlivdef.livery_names) do
		local def = atlivdef.liveries[dname]
		if def and def.livery_template_name then
			local tname = def.livery_template_name
			local t = livgroups[tname]
			if not t then
				t = {}
				livgroups[tname] = t
			end
			local act = "alt:" .. k
			table.insert(t, {dname, act})
		end
	end
	for _, tname in ipairs(atlivdef.template_names) do
		local tt = livgroups[tname] or {}
		local tdef = atlivdef.templates[tname]
		table.insert(desctext, S("@1 by @2: @3",
			H.bold(tname), H.escape(tdef.designer or S "Unknown author"), H.escape(tdef.notes or "")))
		for _, liv in ipairs(tt) do
			local dname, act = unpack(liv)
			local lnht = H.escape(dname)
			if act == livsel then
				lnht = H.bold(lnht, true)
			end
			table.insert(desctext, H.listitem(H.action("preview:" .. act, lnht, true), true))
		end
	end
	return desctext
end)

local function doc_render_wagon_information(prototype, pname)
	local tabopen = wtabopen[pname]
	if not wtabs[tabopen] then
		tabopen = "summary"
		wtabopen[pname] = tabopen
	end
	local fs = wtabs[tabopen](prototype, pname)
	if not fs or fs == "" then
		tabopen = "summary"
		wtabopen[pname], fs = tabopen, wtabs.summary(prototype, pname)
	end
	local ht = {
		("<item name=%s width=40 height=40>"):format(H.escape(prototype.name)),
	}
	for _, v in ipairs {
		{"summary", S("Summary"), true},
		{"capacity", S("Wagon capacity"), true},
		{"bytecode", S("Callback functions"), true},
		{"liv_dlx", S("Dlxtrains liveries"), prototype.dlxtrains_livery},
		{"liv_mul", S("Multi-component liveries"), prototype.livery_definition},
		{"liv_alt", S("Advtrains livery tools (Marnack)"), prototype.advtrains_livery_tools},
	} do
		if v[3] then
			local ent = H.action(v[1], v[2])
			if v[1] == tabopen then
				ent = H.bold(v[2])
			end
			table.insert(ht, H.listitem(ent, true))
		end
	end

	return fs .. ("hypertext[%f,%f;%f,%f;navigation;%s]"):format(docfsdim.nx0, docfsdim.ny0, docfsdim.nw, docfsdim.nh, fsescape(table.concat(ht, "\n")))
end

if doc then
	advtrains_doc_integration.register_on_prototype_loaded(function(itemname, prototype)
		minetest.override_item(itemname, {_doc_items_create_entry = false})
		doc.add_entry("advtrains_wagons", itemname, {
			name = ItemStack(itemname):get_short_description(),
			data = prototype,
		})
		if doc.sub.identifier then
			doc.sub.identifier.register_object(itemname, "advtrains_wagons", itemname)
		end
	end)

	if doc.sub.items then
		local register_factoid = doc.sub.items.register_factoid
		local function ndef_field_factoid(cat, ftype, f, ...)
			local ftp = type(f)
			if ftp == "string" then
				local desc = f
				f = function(x)
					if x ~= nil then
						return desc
					end
				end
			end
			local keys = {...}
			local idx = function(t)
				for _, k in ipairs(keys) do
					if type(t) ~= "table" then
						return nil
					end
					t = t[k]
				end
				return t
			end
			local function fgen(_, def)
				return f(idx(def)) or ""
			end
			return register_factoid(cat, ftype, fgen)
		end
		local function group_factoid(cat, gr, f)
			local function func(x)
				return f(x or 0)
			end
			return ndef_field_factoid(cat, "groups", func, "groups", gr)
		end
		for cat, cinfo in pairs{
			nodes = {
				not_blocking_trains = S("This block does not block trains."),
				save_in_at_nodedb = S("This block is saved in the Advtrains node database."),
			},
		} do
			for group, ginfo in pairs(cinfo) do
				local tp = type(ginfo)
				if tp == "string" then
					group_factoid(cat, group, function(x)
						if x > 0 then
							return ginfo
						end
					end)
				elseif tp == "function" then
					group_factoid(cat, group, ginfo)
				end
			end
		end
		for fname, t in pairs {
			advtrains = {
				on_train_enter = S("This track reacts to passing trains."),
				on_train_approach = S("This track reacts to approaching trains."),
			},
			luaautomation = {
				fire_event = S("This block handles LuaATC events."),
			},
		} do
			for subfname, val in pairs(t) do
				local function f(x)
					if x ~= nil then
						return val
					end
				end
				ndef_field_factoid("nodes", "groups", f, fname, subfname)
			end
		end
		register_factoid("nodes", "groups", function(_, ndef)
			if ndef.advtrains then
				local atdef = ndef.advtrains
				if atdef.set_aspect then
					return S("This is a signal with a variable aspect.")
				elseif atdef.get_aspect then
					return S("This is a signal with a static aspect.")
				end
			end
			if ndef.at_conns then
				return S("This track has the following conns table by default: @1", D.conns(ndef.at_conns) or "?")
			end
			return ""
		end)
	end

	doc.add_category("advtrains_wagons", {
		name = S("Wagons"),
		build_formspec = doc_render_wagon_information,
	})
end

local livtabs = { liv_dlx = true, liv_mul = true, liv_alt = true }
minetest.register_on_player_receive_fields(function(player, formname, fields)
	if formname ~= "doc:entry" then
		return
	end
	local pname = player:get_player_name()
	local cat, ent = doc.get_selection(pname)
	if cat ~= "advtrains_wagons" or ent == nil then
		return
	end

	if fields.wagon_reset_preview then
		set_livery_preview_selection(pname, ent, nil)
		doc.show_entry(pname, cat, ent)
		return true
	end

	local tab = fields.navigation
	if tab and string.sub(tab, 1, 7) == "action:" then
		wtabopen[pname] = string.sub(tab, 8)
		doc.show_entry(pname, cat, ent)
		return true
	end

	local act = fields.entry_body
	if not act then
		return
	end
	local prototype = advtrains_doc_integration.prototypes[ent]
	local tabopen = wtabopen[pname]
	if tabopen == "summary" then
		local sounds = {
			["action:playhorn"] = prototype.horn_sound,
			["action:playdooropen"] = prototype.doors.open.sound,
			["action:playdoorclose"] = prototype.doors.close.sound,
		}
		if sounds[act] then
			minetest.sound_play(sounds[act], {to_player = pname}, true)
		end
	elseif livtabs[tabopen] then
		local txid = string.match(act, [[^action:preview:(.+)$]])
		if txid then
			set_livery_preview_selection(pname, ent, txid)
			doc.show_entry(pname, cat, ent)
		end
	end
end)
