--- Describing things.
-- This module includes functions used for describing things.
-- @module advtrains_doc_integration.describe
-- @alias D
local M = advtrains_doc_integration.mathutils
local utils = advtrains_doc_integration.utils
local D = {}

--- Describe a conns object
-- @tparam advtrains.conns|{advtrains.conns,number,...} conns The conns object to describe.
-- @treturn string|nil The description of the conns object.
function D.conns(conns)
	local connsdesc = {[0] = "N", "NNE", "NE", "ENE", "E", "ESE", "SE", "SSE", "S", "SSW", "SW", "WSW", "W", "WNW", "NW", "NNW"}
	if type(conns) == "table" then
		if conns.c then
			if conns.y and conns.y ~= 0 then
				return ("%s%+d%%"):format(D.conns(conns.c), conns.y*100)
			else
				return D.conns(conns.c)
			end
		else
			local cst = utils.map(conns, D.conns)
			local cstl = #cst
			if cstl == 2 then
				return ("%s - %s"):format(unpack(cst))
			elseif cstl == 3 then
				return ("[%s <-] %s -> %s"):format(cst[3], cst[1], cst[2])
			elseif cstl == 4 then
				return ("%s - %s; %s - %s"):format(unpack(cst))
			elseif cstl == 5 then
				return ("[%s,%s <-] %s -> %s"):format(cst[3], cst[4], cst[1], cst[2])
			end
		end
	else
		return connsdesc[tonumber(conns)]
	end
end

--- Describe a (mixed) fraction
-- @tparam integer int The integer part of the number.
-- @tparam integer num The numerator.
-- @tparam integer denom The denominator.
-- @treturn string The description of the mixed fraction.
function D.mixed_fraction(int, num, denom)
	local str = tostring(int)
	if num > 0 then
		local fs = ("%d/%d"):format(num, denom)
		if int ~= 0 then
			str = ("%d %s"):format(int, fs)
		else
			str = fs
		end
	end
	return str
end

--- Describe a short length (using mm and in).
-- @tparam number x The length in meters.
-- @treturn string The description of the length.
function D.length(x)
	local inch, inchfrac = math.modf(x/0.0254)
	local ft = math.floor(inch/12)
	inch, inchfrac = inch%12, math.floor(inchfrac*32)
	local imst = {}
	if ft > 0 then
		table.insert(imst, ft .. "'")
	end
	if inch > 0 or inchfrac > 0 or ft == 0 then
		table.insert(imst, D.mixed_fraction(inch, M.reduce_fraction(inchfrac, 32)) .. '"')
	end
	return ("%d mm (%s)"):format(1000*x, table.concat(imst, " "))
end

--- Describe a speed value (using m/s, km/h and mph).
-- @tparam number x The speed in m/s.
-- @treturn string The description of the speed.
function D.speed(x)
	local kmph = x*3.6
	local mph = kmph/1.609344
	return string.format("%.1f m/s (%.1f km/h; %.1f mph)", x, kmph, mph)
end

return D
