---@meta _
-- DRAFT 1 DONE
-- luanti/doc/lua_api.md: 'core' namespace reference > Authentication

--[[
* `core.string_to_privs(str[, delim])`:
    * Converts string representation of privs into table form
    * `delim`: String separating the privs. Defaults to `","`.
    * Returns `{ priv1 = true, ... }`
]]
---@nodiscard
---@param str string
---@param delim string?
---@return core.PrivilegeSet
function core.string_to_privs(str, delim) end

--[[
* `core.privs_to_string(privs[, delim])`:
    * Returns the string representation of `privs`
    * `delim`: String to delimit privs. Defaults to `","`.
]]
---@nodiscard
---@param privs core.PrivilegeSet
---@param delim string?
---@return string
function core.privs_to_string(privs, delim) end

--[[
* `core.get_player_privs(name) -> {priv1=true,...}`

`core.set_player_password`, `core.set_player_privs`,
`core.get_player_privs` and `core.auth_reload` call the authentication
handler.
]]
---@nodiscard
---@param name string
---@return core.PrivilegeSet
function core.get_player_privs(name) end

--[[
* `core.check_player_privs(player_or_name, ...)`:
  returns `bool, missing_privs`
    * A quickhand for checking privileges.
    * `player_or_name`: Either a Player object or the name of a player.
    * `...` is either a list of strings, e.g. `"priva", "privb"` or
      a table, e.g. `{ priva = true, privb = true }`.
]]
---@nodiscard
---@param player_or_name core.PlayerRef|string
---@param privs core.PrivilegeSet
---@return boolean, core.PrivilegeSet missing_privs
function core.check_player_privs(player_or_name, privs) end

--[[
* `core.check_player_privs(player_or_name, ...)`:
  returns `bool, missing_privs`
    * A quickhand for checking privileges.
    * `player_or_name`: Either a Player object or the name of a player.
    * `...` is either a list of strings, e.g. `"priva", "privb"` or
      a table, e.g. `{ priva = true, privb = true }`.
]]
---@nodiscard
---@param player_or_name core.PlayerRef|string
---@param ... core.PrivilegeSet.keys
---@return boolean, core.PrivilegeSet missing_privs
function core.check_player_privs(player_or_name, ...) end

--[[
* `core.check_password_entry(name, entry, password)`
    * Returns true if the "password entry" for a player with name matches given
      password, false otherwise.
    * The "password entry" is the password representation generated by the
      engine as returned as part of a `get_auth()` call on the auth handler.
    * Only use this function for making it possible to log in via password from
      external protocols such as IRC, other uses are frowned upon.
]]
---@param name string
---@param entry string
---@param password string
---@return boolean
function core.check_password_entry(name, entry, password) end

--[[
* `core.get_password_hash(name, raw_password)`
    * Convert a name-password pair to a password hash that Luanti can use.
    * The returned value alone is not a good basis for password checks based
      on comparing the password hash in the database with the password hash
      from the function, with an externally provided password, as the hash
      in the db might use the new SRP verifier format.
    * For this purpose, use `core.check_password_entry` instead.
]]
---@nodiscard
---@param name string
---@param raw_password string
---@return string
function core.get_password_hash(name, raw_password) end

--[[
* `core.get_player_ip(name)`: returns an IP address string for the player
  `name`.
    * The player needs to be online for this to be successful.
]]
---@nodiscard
---@param name string
---@return string?
function core.get_player_ip(name) end

--[[
* `core.get_auth_handler()`: Return the currently active auth handler
    * Must be called *after* load time, to ensure that any custom auth handler was
      already registered.
    * See the [Authentication handler definition](#authentication-handler-definition)
    * Use this to e.g. get the authentication data for a player:
      `local auth_data = core.get_auth_handler().get_auth(playername)`
]]
---@nodiscard
---@return core.AuthenticationHandlerDef
function core.get_auth_handler() end

--[[
* `core.notify_authentication_modified(name)`
    * Must be called by the authentication handler for privilege changes.
    * `name`: string; if omitted, all auth data should be considered modified
]]
---@param name string?
function core.notify_authentication_modified(name) end

--[[
* `core.set_player_password(name, password_hash)`: Set password hash of
  player `name`.

`core.set_player_password`, `core.set_player_privs`,
`core.get_player_privs` and `core.auth_reload` call the authentication
handler.
]]
---@param name string
---@param password_hash string
function core.set_player_password(name, password_hash) end

--[[
* `core.set_player_privs(name, privs)`: Set privileges of player `name`.
    * `privs` is a **set** of privileges:
      A table where the keys are names of privileges and the values are `true`.
    * Example: `core.set_player_privs("singleplayer", {interact = true, fly = true})`.
      This **sets** the player privileges to `interact` and `fly`;
      `singleplayer` will only have these two privileges afterwards.

`core.set_player_password`, `core.set_player_privs`,
`core.get_player_privs` and `core.auth_reload` call the authentication
handler.
--]]
---@param name string
---@param privs core.PrivilegeSet
function core.set_player_privs(name, privs) end

--[[
* `core.change_player_privs(name, changes)`: Helper to grant or revoke privileges.
    * `changes`: Table of changes to make.
      A field `[privname] = true` grants a privilege,
      whereas `[privname] = false` revokes a privilege.
    * Example: `core.change_player_privs("singleplayer", {interact = true, fly = false})`
      will grant singleplayer the `interact` privilege
      and revoke singleplayer's `fly` privilege.
      All other privileges will remain unchanged.

**UNOFFICIAL**

For directly replacing all of a player's set of privileges with
another set, see `core.set_player_privs(...)`
]]
---@param name string
---@param changes core.PrivilegeSet
function core.change_player_privs(name, changes) end

--[[
* `core.auth_reload()`
    * See `reload()` in authentication handler definition

`core.set_player_password`, `core.set_player_privs`,
`core.get_player_privs` and `core.auth_reload` call the authentication
handler.
]]
---@return boolean
function core.auth_reload() end