local storage = minetest.get_mod_storage(
)

local S = minetest.get_translator(
    "mobconf"
)

minetest.register_privilege(
    'mobconf',
    {
        description = S(
            "Can add/remove/reconfigure mobs using MobConf"
        ),
        give_to_singleplayer = true,
    }
)

local configured_mobs = {
}

local configured_mobs_stored = storage:get(
    "configured_mobs"
)

if configured_mobs_stored then
    configured_mobs = minetest.deserialize(
        configured_mobs_stored
    )
end

local pending_removals = {
}

local pending_removals_stored = storage:get(
    "pending_removals"
)

if pending_removals_stored then
    pending_removals = minetest.deserialize(
        pending_removals_stored
    )
end

local postpone_removal = function(
    objid,
    name
)
    mtobjid.on_next_activation(
        objid,
        function (
            found
        )
            mobs:remove(
                found:get_luaentity(
                )
            )
            pending_removals[
                objid
            ] = nil
            storage:set_string(
                "pending_removals",
                minetest.serialize(
                    pending_removals
                )
            )
            if name then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "successfully removed mob"
                    )
                )
            end
            return true
        end
    )
end

for pending_removal, _ in pairs(
    pending_removals
) do
    postpone_removal(
        pending_removal
    )
end

local has_simple_dialogs = minetest.get_modpath(
    "simple_dialogs"
)

minetest.register_chatcommand(
    "list_npcs",
    {
        description = S(
            "list configured NPCs"
        ),
        privs = {
            mobconf = true,
        },
        func = function (
            player_name,
            param
        )
            minetest.chat_send_player(
                player_name,
                "mobconf: " .. S(
                    "configured NPCs"
                )
            )
            for id, objid in pairs(
                configured_mobs
            ) do
                local found = mtobjid.get_object_by_id(
                    objid
                )
                local message = "ID " .. id .. ":"
                if found then
                    local ent = found:get_luaentity(
                    )
                    local name = ent.nametag
                    if name then
                        message = message .. " " .. S(
                            "name"
                        ) .. " " .. name
                    else
                        message = message .. " " .. S(
                            "unnamed"
                        )
                    end
                else
                    message = message .. " " .. minetest.colorize(
                        "grey",
                        S(
                            "inactive"
                        )
                    )
                end
                minetest.chat_send_player(
                    player_name,
                    message
                )
            end
        end,
    }
)

minetest.register_chatcommand(
    "add_npc",
    {
        params = "<id> <position> [name] [texture]" .. (
            has_simple_dialogs and " [dialog]" or ""
        ),
        description = has_simple_dialogs and S(
            "Add NPC with ID <id> at <position>, optionally setting nametag [name], texture [texture] and dialog file [dialog]"
        ) or S(
            "Add NPC with ID <id> at <position>, optionally setting nametag [name] and texture [texture]"
        ),
        privs = {
            mobconf = true,
        },
        func = function (
            player_name,
            param
        )
            local arguments = {
            }
            for argument in string.gmatch(
                param,
                "[^ ]+"
            ) do
                arguments[
                    #arguments + 1
                ] = argument
            end
            local max_arguments = has_simple_dialogs and 5 or 4
            if max_arguments < #arguments then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "too many parameters given"
                    )
                )
                return
            end
            if 2 > #arguments then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "too few parameters given"
                    )
                )
                return
            end
            local pos = minetest.string_to_pos(
                arguments[
                    2
                ]
            )
            if not pos then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "could not interpret position"
                    )
                )
                return
            end
            local id = arguments[
                1
            ]
            local name = arguments[
                3
            ]
            local texture = arguments[
                4
            ]
            local dialog_file = arguments[
                5
            ]
            if configured_mobs[
                id
            ] then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "mob with ID @1 already exists",
                        id
                    )
                )
                return
            end
            local definition = {
                name = "mobs_npc:npc",
                ignore_count = true,
            }
            if name then
                definition.nametag = name
            end
            local object = mobs:add_mob(
                pos,
                definition
            )
            if not object then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "could not add mob"
                    )
                )
                return
            end
            local objid = mtobjid.get_id(
                object
            )
            if not objid then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "could not register mob"
                    )
                )
                return
            end
            if texture then
                object.base_texture = {
                    texture,
                }
                object.object:set_properties(
                    {
                        textures = {
                            texture,
                        },
                    }
                )
            end
            object.armor = {
                immortal = 1,
            }
            object.object:set_armor_groups(
                object.armor
            )
            if dialog_file then
                simple_dialogs.load_dialog_from_file(
                    object,
                    "simple_dialogs",
                    dialog_file
                )
            end
            object.order = "stand"
            object.fall_damage = false
            object.attack_players = false
            object.owner = player_name
            minetest.chat_send_player(
                player_name,
                "mobconf: " .. S(
                    "successfully added mob"
                )
            )
            configured_mobs[
                id
            ] = objid
            storage:set_string(
                "configured_mobs",
                minetest.serialize(
                    configured_mobs
                )
            )
        end,
    }
)

minetest.register_chatcommand(
    "remove_npc",
    {
        params = "<id>",
        description = S(
            "Remove NPC with ID <id>"
        ),
        privs = {
            mobconf = true,
        },
        func = function (
            name,
            param
        )
            local id = param
            local objid = configured_mobs[
                id
            ]
            if not objid then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "mob with ID @1 does not exist",
                        id
                    )
                )
                return
            end
            local found = mtobjid.get_object_by_id(
                objid
            )
            if found then
                mobs:remove(
                    found:get_luaentity(
                    )
                )
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "successfully removed mob"
                    )
                )
            else
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "mob could not be found"
                    )
                )
                postpone_removal(
                    objid,
                    name
                )
                pending_removals[
                    objid
                ] = 1
                storage:set_string(
                    "pending_removals",
                    minetest.serialize(
                        pending_removals
                    )
                )
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "postponed action"
                    )
                )
            end
            configured_mobs[
                id
            ] = nil
            storage:set_string(
                "configured_mobs",
                minetest.serialize(
                    configured_mobs
                )
            )
        end,
    }
)

minetest.register_chatcommand(
    "npc_load_dialog",
    {
        params = "<id> <dialog>",
        description = S(
            "Load dialog file <dialog> for NPC with ID <id>"
        ),
        privs = {
            mobconf = true,
        },
        func = function (
            name,
            param
        )
            if not has_simple_dialogs then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "unsupported operation"
                    )
                )
                minetest.log(
                    "warning",
                    "need to install simple_dialogs mod to configure dialogs"
                )
                return
            end
            local arguments = {
            }
            for argument in string.gmatch(
                param,
                "[^ ]+"
            ) do
                arguments[
                    #arguments + 1
                ] = argument
            end
            if 2 < #arguments then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "too many parameters given"
                    )
                )
                return
            end
            if 2 > #arguments then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "too few parameters given"
                    )
                )
                return
            end
            local id = arguments[
                1
            ]
            local dialog_file = arguments[
                2
            ]
            local objid = configured_mobs[
                id
            ]
            if not objid then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "mob with ID @1 does not exist",
                        id
                    )
                )
                return
            end
            local found = mtobjid.get_object_by_id(
                objid
            )
            if not found then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "mob could not be found"
                    )
                )
                return
            end
            simple_dialogs.load_dialog_from_file(
                found:get_luaentity(
                ),
                "simple_dialogs",
                dialog_file
            )
            minetest.chat_send_player(
                name,
                "mobconf: " .. S(
                    "dialog file loaded"
                )
            )
        end,
    }
)

minetest.register_chatcommand(
    "npc_set_name",
    {
        params = "<id> [name]",
        description = S(
            "Set nametag of NPC with ID <id> to [name]"
        ),
        privs = {
            mobconf = true,
        },
        func = function (
            player_name,
            param
        )
            local arguments = {
            }
            for argument in string.gmatch(
                param,
                "[^ ]+"
            ) do
                arguments[
                    #arguments + 1
                ] = argument
            end
            if 2 < #arguments then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "too many parameters given"
                    )
                )
                return
            end
            if 1 > #arguments then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "too few parameters given"
                    )
                )
                return
            end
            local id = arguments[
                1
            ]
            local name = arguments[
                2
            ] or ""
            local objid = configured_mobs[
                id
            ]
            if not objid then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "mob with ID @1 does not exist",
                        id
                    )
                )
                return
            end
            local found = mtobjid.get_object_by_id(
                objid
            )
            if not found then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "mob could not be found"
                    )
                )
                return
            end
            local ent = found:get_luaentity(
            )
            ent.nametag = name
            ent:update_tag(
            )
            minetest.chat_send_player(
                player_name,
                "mobconf: " .. S(
                    "nametag set"
                )
            )
        end,
    }
)

minetest.register_chatcommand(
    "npc_set_texture",
    {
        params = "<id> <texture>",
        description = S(
            "Set texture of NPC with ID <id> to <texture>"
        ),
        privs = {
            mobconf = true,
        },
        func = function (
            name,
            param
        )
            local arguments = {
            }
            for argument in string.gmatch(
                param,
                "[^ ]+"
            ) do
                arguments[
                    #arguments + 1
                ] = argument
            end
            if 2 < #arguments then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "too many parameters given"
                    )
                )
                return
            end
            if 2 > #arguments then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "too few parameters given"
                    )
                )
                return
            end
            local id = arguments[
                1
            ]
            local texture = arguments[
                2
            ]
            local objid = configured_mobs[
                id
            ]
            if not objid then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "mob with ID @1 does not exist",
                        id
                    )
                )
                return
            end
            local found = mtobjid.get_object_by_id(
                objid
            )
            if not found then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "mob could not be found"
                    )
                )
                return
            end
            local ent = found:get_luaentity(
            )
            ent.base_texture = {
                texture,
            }
            found:set_properties(
                {
                    textures = {
                        texture,
                    },
                }
            )
            minetest.chat_send_player(
                name,
                "mobconf: " .. S(
                    "texture set"
                )
            )
        end,
    }
)

minetest.register_chatcommand(
    "npc_set_pos",
    {
        params = "<id> <position>",
        description = S(
            "Set the position of NPC with ID <id> to <position>"
        ),
        privs = {
            mobconf = true,
        },
        func = function (
            name,
            param
        )
            local arguments = {
            }
            for argument in string.gmatch(
                param,
                "[^ ]+"
            ) do
                arguments[
                    #arguments + 1
                ] = argument
            end
            if 2 < #arguments then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "too many parameters given"
                    )
                )
                return
            end
            if 2 > #arguments then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "too few parameters given"
                    )
                )
                return
            end
            local id = arguments[
                1
            ]
            local pos = minetest.string_to_pos(
                arguments[
                    2
                ]
            )
            if not pos then
                minetest.chat_send_player(
                    player_name,
                    "mobconf: " .. S(
                        "could not interpret position"
                    )
                )
                return
            end
            local objid = configured_mobs[
                id
            ]
            if not objid then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "mob with ID @1 does not exist",
                        id
                    )
                )
                return
            end
            local found = mtobjid.get_object_by_id(
                objid
            )
            if not found then
                minetest.chat_send_player(
                    name,
                    "mobconf: " .. S(
                        "mob could not be found"
                    )
                )
                return
            end
            found:set_pos(
                pos
            )
            minetest.chat_send_player(
                name,
                "mobconf: " .. S(
                    "position set"
                )
            )
        end,
    }
)
