local oh = obelisk_analog
local C = oh.C
local random, floor = math.random, math.floor

local function find_surface_pos(x, z)
    for y = 100, -20, -1 do
        local node = minetest.get_node({x = x, y = y, z = z})
        local above = minetest.get_node({x = x, y = y + 1, z = z})
        if node.name ~= "air" and node.name ~= "ignore" and above.name == "air" then
            return {x = x, y = y + 1, z = z}
        end
    end
    return nil
end

local function generate_abandoned_house(pos)
    if oh.build_house_with_validation then
        oh.build_house_with_validation(pos, "overworld")
        minetest.log("action", "[obelisk_analog] Validated house generated at " .. minetest.pos_to_string(pos))
        return
    end

    local width = random(5, 8)
    local depth = random(6, 10)
    local height = random(4, 6)
    local base_y = pos.y

    for x = pos.x, pos.x + width do
        for z = pos.z, pos.z + depth do
            minetest.set_node({x = x, y = base_y, z = z}, {name = "default:wood"})
        end
    end

    for y = base_y + 1, base_y + height do
        for x = pos.x, pos.x + width do
            if random() > 0.1 then
                minetest.set_node({x = x, y = y, z = pos.z}, {name = "default:wood"})
            end
            if random() > 0.1 then
                minetest.set_node({x = x, y = y, z = pos.z + depth}, {name = "default:wood"})
            end
        end
        for z = pos.z, pos.z + depth do
            if random() > 0.1 then
                minetest.set_node({x = pos.x, y = y, z = z}, {name = "default:wood"})
            end
            if random() > 0.1 then
                minetest.set_node({x = pos.x + width, y = y, z = z}, {name = "default:wood"})
            end
        end
    end

    for x = pos.x, pos.x + width do
        for z = pos.z, pos.z + depth do
            if random() > 0.2 then
                minetest.set_node({x = x, y = base_y + height, z = z}, {name = "default:wood"})
            end
        end
    end

    local door_x = pos.x + floor(width / 2)
    minetest.set_node({x = door_x, y = base_y + 1, z = pos.z}, {name = "air"})
    minetest.set_node({x = door_x, y = base_y + 2, z = pos.z}, {name = "air"})

    local window_y = base_y + 2
    if random() > 0.3 then
        minetest.set_node({x = pos.x + 2, y = window_y, z = pos.z + depth}, {name = "default:glass"})
    end
    if random() > 0.3 then
        minetest.set_node({x = pos.x + width - 2, y = window_y, z = pos.z + depth}, {name = "default:glass"})
    end

    local chest_pos = {x = pos.x + width - 1, y = base_y + 1, z = pos.z + depth - 1}
    minetest.set_node(chest_pos, {name = "default:chest"})
    local meta = minetest.get_meta(chest_pos)
    local inv = meta:get_inventory()
    inv:set_size("main", 32)

    local loot = {"default:torch", "default:apple", "default:paper", "default:book", "obelisk_analog:portal_key_sequence"}

    for i = 1, random(2, 4) do
        local item = loot[random(#loot)]
        local count = (item == "obelisk_analog:portal_key_sequence") and 1 or random(1, 5)
        if random() < 0.3 or item ~= "obelisk_analog:portal_key_sequence" then
            inv:add_item("main", item .. " " .. count)
        end
    end

    if random() > 0.5 then
        minetest.set_node({x = pos.x + 1, y = base_y + 2, z = pos.z + 1}, {name = "default:torch_wall", param2 = 4})
    end

    minetest.log("action", "[obelisk_analog] Abandoned house generated at " .. minetest.pos_to_string(pos))
end

local function generate_creepy_shrine(pos)
    local base_y = pos.y

    for x = pos.x - 2, pos.x + 2 do
        for z = pos.z - 2, pos.z + 2 do
            minetest.set_node({x = x, y = base_y, z = z}, {name = "obelisk_analog:shrine_block"})
        end
    end

    local corners = {{-2, -2}, {2, -2}, {-2, 2}, {2, 2}}
    for _, c in ipairs(corners) do
        minetest.set_node({x = pos.x + c[1], y = base_y + 1, z = pos.z + c[2]}, {name = "obelisk_analog:shrine_block"})
        minetest.set_node({x = pos.x + c[1], y = base_y + 2, z = pos.z + c[2]}, {name = "default:torch"})
    end

    minetest.set_node({x = pos.x, y = base_y + 1, z = pos.z}, {name = "obelisk_analog:ritual_stone"})

    minetest.log("action", "[obelisk_analog] Creepy shrine generated at " .. minetest.pos_to_string(pos))
end

local function generate_warning_sign(pos)
    minetest.set_node({x = pos.x, y = pos.y, z = pos.z}, {name = "default:fence_wood"})
    minetest.set_node({x = pos.x, y = pos.y + 1, z = pos.z}, {name = "default:fence_wood"})
    minetest.set_node({x = pos.x, y = pos.y + 2, z = pos.z}, {name = "obelisk_analog:warning_sign", param2 = random(0, 3)})
end

local function generate_ritual_circle(pos)
    local base_y = pos.y
    local radius = 4

    for angle = 0, 2 * math.pi, math.pi / 8 do
        local x = pos.x + floor(math.cos(angle) * radius + 0.5)
        local z = pos.z + floor(math.sin(angle) * radius + 0.5)
        minetest.set_node({x = x, y = base_y, z = z}, {name = "obelisk_analog:ritual_stone"})
    end

    minetest.set_node({x = pos.x, y = base_y, z = pos.z}, {name = "obelisk_analog:portal_activator", param2 = 0})

    for i = 1, 4 do
        local angle = (i - 1) * math.pi / 2
        local x = pos.x + floor(math.cos(angle) * 2 + 0.5)
        local z = pos.z + floor(math.sin(angle) * 2 + 0.5)
        minetest.set_node({x = x, y = base_y + 1, z = z}, {name = "default:torch"})
    end

    minetest.log("action", "[obelisk_analog] Ritual circle generated at " .. minetest.pos_to_string(pos))
end

local function generate_grave_site(pos)
    local base_y = pos.y

    for i = 1, random(3, 6) do
        local gx = pos.x + random(-5, 5)
        local gz = pos.z + random(-5, 5)
        minetest.set_node({x = gx, y = base_y, z = gz}, {name = "default:dirt"})
        minetest.set_node({x = gx, y = base_y + 1, z = gz}, {name = "default:cobble"})
    end

    minetest.set_node({x = pos.x, y = base_y, z = pos.z}, {name = "default:dirt"})
    minetest.set_node({x = pos.x, y = base_y + 1, z = pos.z}, {name = "default:cobble"})
    minetest.set_node({x = pos.x, y = base_y + 2, z = pos.z}, {name = "default:cobble"})
end

local function generate_ruined_tower(pos)
    local base_y = pos.y
    local height = random(8, 15)
    local radius = 3

    for y = base_y, base_y + height do
        local decay = (y - base_y) / height
        for angle = 0, 2 * math.pi, math.pi / 4 do
            if random() > decay * 0.5 then
                local x = pos.x + floor(math.cos(angle) * radius + 0.5)
                local z = pos.z + floor(math.sin(angle) * radius + 0.5)
                minetest.set_node({x = x, y = y, z = z}, {name = "default:cobble"})
            end
        end
    end

    for y = base_y + 1, base_y + height - 2, 3 do
        minetest.set_node({x = pos.x + radius, y = y, z = pos.z}, {name = "air"})
    end

    if random() > 0.5 then
        local chest_y = base_y + height - 3
        minetest.set_node({x = pos.x, y = chest_y, z = pos.z}, {name = "default:chest"})
        local meta = minetest.get_meta({x = pos.x, y = chest_y, z = pos.z})
        local inv = meta:get_inventory()
        inv:set_size("main", 32)
        inv:add_item("main", "obelisk_analog:portal_key_sequence")
    end

    minetest.log("action", "[obelisk_analog] Ruined tower generated at " .. minetest.pos_to_string(pos))
end

local function generate_creepy_well(pos)
    local base_y = pos.y

    for x = pos.x - 1, pos.x + 1 do
        for z = pos.z - 1, pos.z + 1 do
            if not (x == pos.x and z == pos.z) then
                minetest.set_node({x = x, y = base_y, z = z}, {name = "default:cobble"})
                minetest.set_node({x = x, y = base_y + 1, z = z}, {name = "default:cobble"})
            end
        end
    end

    for y = base_y - 10, base_y - 1 do
        minetest.set_node({x = pos.x, y = y, z = pos.z}, {name = "air"})
    end

    minetest.set_node({x = pos.x - 1, y = base_y + 2, z = pos.z}, {name = "default:fence_wood"})
    minetest.set_node({x = pos.x + 1, y = base_y + 2, z = pos.z}, {name = "default:fence_wood"})
    minetest.set_node({x = pos.x - 1, y = base_y + 3, z = pos.z}, {name = "default:wood"})
    minetest.set_node({x = pos.x, y = base_y + 3, z = pos.z}, {name = "default:wood"})
    minetest.set_node({x = pos.x + 1, y = base_y + 3, z = pos.z}, {name = "default:wood"})
end

minetest.register_on_generated(function(minp, maxp, blockseed)
    if minp.y > 100 or maxp.y < -20 then return end
    if oh.is_in_dimension(minp) then return end

    math.randomseed(blockseed)

    local structures = {
        {chance = C.STRUCTURE.HOUSE_SPAWN_CHANCE, func = generate_abandoned_house, margin = 10},
        {chance = C.STRUCTURE.SHRINE_SPAWN_CHANCE, func = generate_creepy_shrine, margin = 5},
        {chance = C.STRUCTURE.SIGN_SPAWN_CHANCE, func = generate_warning_sign, margin = 0},
        {chance = C.STRUCTURE.RITUAL_SPAWN_CHANCE, func = generate_ritual_circle, margin = 10},
        {chance = C.STRUCTURE.GRAVE_SPAWN_CHANCE, func = generate_grave_site, margin = 5},
        {chance = C.STRUCTURE.TOWER_SPAWN_CHANCE, func = generate_ruined_tower, margin = 5},
        {chance = C.STRUCTURE.WELL_SPAWN_CHANCE, func = generate_creepy_well, margin = 3},
    }

    for _, struct in ipairs(structures) do
        if random() < struct.chance then
            local x = random(minp.x + struct.margin, maxp.x - struct.margin)
            local z = random(minp.z + struct.margin, maxp.z - struct.margin)
            local pos = find_surface_pos(x, z)
            if pos then
                struct.func(pos)
            end
        end
    end
end)

minetest.register_chatcommand("horror_spawn_house", {
    privs = {server = true},
    description = "Spawn an abandoned house at your position",
    func = function(name)
        local player = minetest.get_player_by_name(name)
        if player then
            local pos = player:get_pos()
            pos.y = floor(pos.y)
            generate_abandoned_house(pos)
            return true, "Abandoned house spawned"
        end
        return false, "Player not found"
    end,
})

minetest.register_chatcommand("horror_spawn_shrine", {
    privs = {server = true},
    description = "Spawn a creepy shrine at your position",
    func = function(name)
        local player = minetest.get_player_by_name(name)
        if player then
            local pos = player:get_pos()
            pos.y = floor(pos.y)
            generate_creepy_shrine(pos)
            return true, "Creepy shrine spawned"
        end
        return false, "Player not found"
    end,
})

minetest.register_chatcommand("horror_spawn_ritual", {
    privs = {server = true},
    description = "Spawn a ritual circle at your position",
    func = function(name)
        local player = minetest.get_player_by_name(name)
        if player then
            local pos = player:get_pos()
            pos.y = floor(pos.y)
            generate_ritual_circle(pos)
            return true, "Ritual circle spawned"
        end
        return false, "Player not found"
    end,
})

minetest.register_chatcommand("horror_spawn_tower", {
    privs = {server = true},
    description = "Spawn a ruined tower at your position",
    func = function(name)
        local player = minetest.get_player_by_name(name)
        if player then
            local pos = player:get_pos()
            pos.y = floor(pos.y)
            generate_ruined_tower(pos)
            return true, "Ruined tower spawned"
        end
        return false, "Player not found"
    end,
})
