-- Copyright (C) 2025 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- This is a class that implements the same methods as Env. but when accessing function values it will
-- call them instead, and return the results. Its also read only

local helpers = require("helpers")

local EnvStud =
{
	instance_of = "Env" -- might cause issues with logic that checks whether an instance is an "Env", but whatever
}

EnvStud.__index = EnvStud

function EnvStud:new(t)
	t = t or {}
	self = setmetatable(
	{
		parameters = t.parameters or {},

		inherited_env = t.inherited_env,

		-- this will always be empty, its to fool logic of the Env class so it dosen't think any variables are set
		parameter_attributes = {}
	}, EnvStud)

	return self
end

-- this is the only way to set the parameters for the EnvStud
function EnvStud:special_set(key, value)
	self.parameter_attributes[key] = bit.bor(helpers.ParamAttributes.ReadOnly, helpers.ParamAttributes.Volatile)
	self.parameters[key] = value
end

function EnvStud:get(key)
	-- {{{ type checking
	if key == nil then return nil end
	assert(type(key) == "string", "key should be of type string. instead of " .. type(key))
	-- }}}

	local output = self.parameters[key]

	if output == nil then
		return self.inherited_env and self.inherited_env:get(key)
	end

	if type(output) == "function" then
		output = output(self)
	end

	return output
end

function EnvStud:set(key, value, read_only, is_volatile)
	-- {{{ type checking
	assert(type(key) == "string", "key should be of type string. instead of " .. type(key))
	-- }}}

	if self.inherited_env then
		assert(not self.parameters[key], "trying to overwrite a readonly value")
		self.inherited_env:set(key, value, read_only, is_volatile)
	else
		error("EnvStud dosen't have any inherited_env set")
	end
end

function EnvStud:unset(key)
	if self.inherited_env then
		self.inherited_env:unset(key)
	end
end

function EnvStud:is_param_readonly(key)
	if self.parameters[key] then
		return true
	elseif self.inherited_env then
		return self.inherited_env:is_param_readonly(key)
	else
		error("EnvStud dosen't have any inherited_env set")
	end
end

function EnvStud:inherit(env)
	-- {{{ type checking
	assert(env.instance_of == "Env", "env should be an instance of Env. instead of " .. env.instance_of)
	-- }}}

	self.inherited_env = env
end

return EnvStud
