-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local Expression = require("language_constructs.expression")
local StringExpression = require("language_constructs.string_expression")
local Shell = require("language_constructs.shell")
local ShellContext = require("shell_context")
local DatatypeValidator = require("datatype_validator")

local ForLoop =
{
	instance_of = "LanguageConstruct_ForLoop",
}

ForLoop.__index = ForLoop

function ForLoop:new(t)
	t = t or {}
	self = setmetatable(
	{
		param_name = t.param_name or "",
		iterable = t.iterable or "",
		subshell = t.subshell or "",
	}, ForLoop)

	return self
end


function ForLoop:evaluate(ctx, max_frustration)
	-- {{{ type checking
	assert(ctx.instance_of == "ShellContext", "ctx should be an instance of ShellContext. instead of " .. ctx.instance_of)
	-- }}}

	max_frustration = max_frustration or 10000
	local frustration = 0
	local evaluated_param_name = self.param_name:evaluate(ctx)
	local evaluated_iterable = self.iterable:evaluate(ctx)

	local subshell_ctx = ShellContext:new()
	subshell_ctx.privilege_cache = ctx.privilege_cache

	local iterable_type = DatatypeValidator.get_type_of(evaluated_iterable)
	local iterator

	if type(evaluated_iterable) == "table" and evaluated_iterable.instance_of == "TypedList" then
		local iter_value
		local iter_idx
		iterator = function()
			iter_idx, iter_value = next(evaluated_iterable.container, iter_idx)
			return iter_value
		end
	else
		 iterator = DatatypeValidator.registered_types[iterable_type].iterator(evaluated_iterable)
	end

	ctx:interrupt()

	local function iterate()
		local value = iterator()

		if value == nil then
			ctx:stop_interruption()
			return
		end

		assert(frustration <= max_frustration, "loop exceeded maximum iterations, as defined by lush_max_loop_iterations setting")

		subshell_ctx.env:set(evaluated_param_name, value, false, true)
		self.subshell:evaluate(ctx,
		{
			subshell_context = subshell_ctx,
			complete_callback = function()
				subshell_ctx.stdout:clear()

				assert(ctx.exit_status == 0, "for loop: " .. ctx.stderr)

				frustration = frustration + 1

				iterate()
			end
		})
	end

	iterate()
end

local white_chars_trie = helpers.create_trie({" ", "\t"})
local shell_terminator_trie = helpers.create_trie({"end"})

function ForLoop:parse(parser_ctx)
	-- {{{ type checking
	assert(parser_ctx.instance_of == "ParserContext", "parser_ctx should be an instance of ParserContext. instead of " .. parser_ctx.instance_of)
	-- }}}

	-- state: *

	local success = parser_ctx:match_fast("for")

	assert(success, "expected 'for'")

	-- state: for*

	parser_ctx:consume_whitespaces()

	self.param_name = StringExpression:new()
	self.param_name:parse(parser_ctx, white_chars_trie)

	-- state: for <param_name>*

	parser_ctx:skip_until(helpers.white_chars_set, true)

	success = parser_ctx:match_fast("in")

	assert(success, "expected 'in'")

	-- state: for <param_name> in*

	parser_ctx:consume_whitespaces()

	self.iterable = Expression:new()
	self.iterable:parse(parser_ctx, white_chars_trie)

	-- state: for <param_name> in <iterable>*

	parser_ctx:consume_whitespaces()

	success = parser_ctx:match_fast("do")

	assert(success, "expected 'do'")

	-- state: for <param_name> in <iterable> do*

	parser_ctx:consume_whitespaces()

	self.subshell = Shell:new()
	self.subshell:parse(parser_ctx, {}, shell_terminator_trie)

	parser_ctx:advance(3)
	parser_ctx:consume_whitespaces()

	-- state: for <param_name> in <iterable> do
	-- <subshell>
	-- end*
end

function ForLoop:dump(dump_ctx)
	dump_ctx:write_text(dump_ctx:color("(ForLoop)", "ConstructSpecifier"))
	dump_ctx:new_line()
	dump_ctx:write_text("[")
	dump_ctx:indent(1)
	dump_ctx:new_line()

	dump_ctx:write_text("Iterator name: ")
	self.param_name:dump(dump_ctx)
	dump_ctx:new_line()

	dump_ctx:write_text("Iterable: ")
	self.iterable:dump(dump_ctx)
	dump_ctx:new_line()

	self.subshell:dump(dump_ctx)

	dump_ctx:indent(-1)
	dump_ctx:new_line()
	dump_ctx:write_text("]")
end

return ForLoop
