-- Copyright (C) 2025 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")
require("tests/command_definitions")

describe("LanguageConstruct_FunctionDefinition class: ",
function()
	local FunctionDefinition = require("language_constructs.function_definition")
	local ShellContext = require("shell_context")
	local ParserContext = require("parser_context")
	local helpers = require("helpers")
	local shell_context_instance
	local parser_context_instance
	local function_definition_instance

	before_each(
	function()
		shell_context_instance = ShellContext:new()
		shell_context_instance.env:inherit(lush.global_env)
		parser_context_instance = ParserContext:new()

		shell_context_instance.env:set("parameter", "value")

		function_definition_instance = FunctionDefinition:new()
	end)

	describe("evaluates correctly: ",
	function()
		it("simplest case",
		function()
			parser_context_instance.text = "function() echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			function_definition_instance:evaluate(shell_context_instance)

			function_definition_instance(shell_context_instance, {})

			assert.equals("hi", shell_context_instance.stdout:get(1))
		end)

		it("when passed arguments",
		function()
			parser_context_instance.text = "function(arg1, arg2) echo $arg1; echo $arg2; end"

			function_definition_instance:parse(parser_context_instance)

			function_definition_instance:evaluate(shell_context_instance)

			function_definition_instance(shell_context_instance, {1, 2})

			assert.equals(1, shell_context_instance.stdout:get(1))
			assert.equals(2, shell_context_instance.stdout:get(2))
		end)

		it("when passed optional arguments",
		function()
			parser_context_instance.text = "function(arg1, arg2?) echo $arg1; echo $arg2; end"

			function_definition_instance:parse(parser_context_instance)

			function_definition_instance:evaluate(shell_context_instance)

			function_definition_instance(shell_context_instance, {"1"})

			assert.equals("1", shell_context_instance.stdout:get(1))
			assert.equals(nil, shell_context_instance.stdout:get(2))
		end)

		it("when passed arguments with types",
		function()
			parser_context_instance.text = "function(string arg1, string arg2) echo $arg1; echo $arg2; end"

			function_definition_instance:parse(parser_context_instance)

			function_definition_instance:evaluate(shell_context_instance)

			function_definition_instance(shell_context_instance, {1, "LOL"})

			assert.equals("1", shell_context_instance.stdout:get(1))
			assert.equals("LOL", shell_context_instance.stdout:get(2))
		end)

		it("when passed block arguments",
		function()
			parser_context_instance.text = "function(string arg1...) echo $arg1; end"

			function_definition_instance:parse(parser_context_instance)

			function_definition_instance:evaluate(shell_context_instance)

			function_definition_instance(shell_context_instance, {"hello", 123, true})

			assert.equals("hello", shell_context_instance.stdout:get(1))
			assert.equals("123", shell_context_instance.stdout:get(2))
			assert.equals("true", shell_context_instance.stdout:get(3))
		end)
	end)

	describe("parses correctly: ",
	function()
		it("simplest case",
		function()
			parser_context_instance.text = "function func() echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals("func", function_definition_instance.identifier)
			assert.equals(nil, function_definition_instance.arg_rules)

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("with arguments",
		function()
			parser_context_instance.text = "function func(arg1, arg2) echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals("func", function_definition_instance.identifier)
			assert.same({rule_type="value", name="arg1", single = true}, function_definition_instance.arg_rules[1])
			assert.same({rule_type="value", name="arg2", single = true}, function_definition_instance.arg_rules[2])

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("with arguments that have a type specified",
		function()
			parser_context_instance.text = "function func(string arg1, number arg2) echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals("func", function_definition_instance.identifier)
			assert.same({rule_type="value", name="arg1", type = "string", single = true}, function_definition_instance.arg_rules[1])
			assert.same({rule_type="value", name="arg2", type= "number", single = true}, function_definition_instance.arg_rules[2])

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("with arguments that are optional",
		function()
			parser_context_instance.text = "function func(arg1, arg2?) echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals("func", function_definition_instance.identifier)
			assert.same({rule_type="value", name="arg1", single = true}, function_definition_instance.arg_rules[1])
			assert.same({rule_type="value", name="arg2", optional=true, single = true}, function_definition_instance.arg_rules[2])

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("with arguments that are blocks",
		function()
			parser_context_instance.text = "function func(arg1, arg2...) echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals("func", function_definition_instance.identifier)
			assert.same({rule_type="value", name="arg1", single = true}, function_definition_instance.arg_rules[1])
			assert.same({rule_type="block", name="arg2", single = true}, function_definition_instance.arg_rules[2])

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("with arguments that are blocks and specify type",
		function()
			parser_context_instance.text = "function func(arg1, string arg2...) echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals("func", function_definition_instance.identifier)
			assert.same({rule_type="value", name="arg1", single = true}, function_definition_instance.arg_rules[1])
			assert.same({rule_type="block", name="arg2", type = "string", single = true}, function_definition_instance.arg_rules[2])

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("with arguments that are arrays",
		function()
			parser_context_instance.text = "function func(arg1, string[] arg2) echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals("func", function_definition_instance.identifier)
			assert.same({rule_type="value", name="arg1", single = true}, function_definition_instance.arg_rules[1])
			assert.same({rule_type="value", name="arg2", type = "string", single = false}, function_definition_instance.arg_rules[2])

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("function without a name",
		function()
			parser_context_instance.text = "function() echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals(nil, function_definition_instance.identifier)

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("function without a name, with arguments",
		function()
			parser_context_instance.text = "function(arg1, string arg2, arg3?) echo hi; end"

			function_definition_instance:parse(parser_context_instance)

			assert.equals(nil, function_definition_instance.identifier)
			assert.same({rule_type="value", name="arg1", single = true}, function_definition_instance.arg_rules[1])
			assert.same({rule_type="value", name="arg2", type = "string", single = true}, function_definition_instance.arg_rules[2])
			assert.same({rule_type="value", name="arg3", single = true, optional = true}, function_definition_instance.arg_rules[3])

			-- Should have 2 elements. One for the command, one for the execution operator at the end
			assert.equals(2, #function_definition_instance.shell.actions)
		end)

		it("errors when parameters lack an identifier",
		function()
			parser_context_instance.text = "function func(string[]) echo hi; end"

			assert.errors(function()
				function_definition_instance:parse(parser_context_instance)
			end)
		end)

		it("errors when an odd character is added to the parameter",
		function()
			parser_context_instance.text = "function func(string!) echo hi; end"

			assert.errors(function()
				function_definition_instance:parse(parser_context_instance)
			end)
		end)

		it("errors when given an invalid identifier",
		function()
			parser_context_instance.text = "function func(1arg) echo hi; end"

			assert.errors(function()
				function_definition_instance:parse(parser_context_instance)
			end)
		end)
	end)

end)
