-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")

describe("LanguageConstructs_ParameterAsignment class: ",
function()
	local ParameterAsignment = require("language_constructs.parameter_asignment")
	local ParserContext = require("parser_context")
	local ShellContext = require("shell_context")
	local helpers = require("helpers")
	local parser_context_instance
	local shell_context_instance
	local parameter_asignment_instance
	local terminators = helpers.create_trie({" ", "\t"})

	before_each(
	function()
		parser_context_instance = ParserContext:new()
		shell_context_instance = ShellContext:new()
		parameter_asignment_instance = ParameterAsignment:new()
	end)

	it("parses correctly when given a simple asignment",
	function()
		parser_context_instance.text = "$parameter=value"

		parameter_asignment_instance:parse(parser_context_instance, terminators)
		parameter_asignment_instance:evaluate(shell_context_instance)

		assert.equals(#parser_context_instance.text + 1, parser_context_instance.character_index)
		assert.equals("value", shell_context_instance.env:get("parameter"))
	end)

	it("parses correctly when given a simple asignment and has been already partialy parsed",
	function()
		parser_context_instance.text = "$parameter=value"

		local Expression = require("language_constructs.expression")
		local expression_instance = Expression:new()
		expression_instance:parse(parser_context_instance, helpers.create_trie({"="}))

		parameter_asignment_instance.lvalue = expression_instance

		parameter_asignment_instance:parse(parser_context_instance, terminators)
		parameter_asignment_instance:evaluate(shell_context_instance)

		assert.equals(#parser_context_instance.text + 1, parser_context_instance.character_index)
		assert.equals("value", shell_context_instance.env:get("parameter"))
	end)

	it("parses correctly when given an lvalue with the [] operator",
	function()
		parser_context_instance.text = [[$parameter[2]="value"]]

		shell_context_instance.env:set("parameter", {})

		parameter_asignment_instance:parse(parser_context_instance, terminators)
		parameter_asignment_instance:evaluate(shell_context_instance)

		assert.equals(#parser_context_instance.text + 1, parser_context_instance.character_index)
		assert.same({[2] = "value"}, shell_context_instance.env:get("parameter"))
	end)

	it("parses correctly when given an lvalue with the [] operator",
	function()
		parser_context_instance.text = [[$parameter.member="value"]]

		shell_context_instance.env:set("parameter", {})

		parameter_asignment_instance:parse(parser_context_instance, terminators)
		parameter_asignment_instance:evaluate(shell_context_instance)

		assert.equals(#parser_context_instance.text + 1, parser_context_instance.character_index)
		assert.same({member = "value"}, shell_context_instance.env:get("parameter"))
	end)

	it("parses correctly when given an lvalue with two . operators",
	function()
		parser_context_instance.text = [[$parameter.member.member2="value"]]

		shell_context_instance.env:set("parameter", {member = {}})

		parameter_asignment_instance:parse(parser_context_instance, terminators)
		parameter_asignment_instance:evaluate(shell_context_instance)

		assert.equals(#parser_context_instance.text + 1, parser_context_instance.character_index)
		assert.same({member = {member2 = "value"}}, shell_context_instance.env:get("parameter"))
	end)

	it("parses correctly when given an lvalue with two [] operators",
	function()
		parser_context_instance.text = [[$parameter[1][2]="value"]]

		shell_context_instance.env:set("parameter", {{}})

		parameter_asignment_instance:parse(parser_context_instance, terminators)
		parameter_asignment_instance:evaluate(shell_context_instance)

		assert.equals(#parser_context_instance.text + 1, parser_context_instance.character_index)
		assert.same({{[2] = "value"}}, shell_context_instance.env:get("parameter"))
	end)
end)
