-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")

local helpers = require("helpers")

describe("ParserContext class: ",
function()
	local ParserContext = require("parser_context")
	local parser_context_instance
	local char

	before_each(
	function()
		parser_context_instance = ParserContext:new()
	end)

	it("consuming a character works",
	function()
		parser_context_instance.text = "string"

		char = parser_context_instance:consume()

		assert.equals("s", char)
		assert.equals(2, parser_context_instance.character_index)
	end)

	it("consuming the last character returns an empty string",
	function()
		parser_context_instance.text = ""

		char = parser_context_instance:consume()

		assert.equals("", char)
	end)

	it("consuming multiple characters works",
	function()
		parser_context_instance.text = "string"

		char = parser_context_instance:consume(2)

		assert.equals("t", char)
		assert.equals(3, parser_context_instance.character_index)
	end)

	it("peeking a character works",
	function()
		parser_context_instance.text = "string"

		char = parser_context_instance:peek()

		assert.equals("s", char)
		assert.equals(1, parser_context_instance.character_index)
	end)

	it("peeking the last character returns an empty string",
	function()
		parser_context_instance.text = ""

		char = parser_context_instance:peek()

		assert.equals("", char)
	end)

	it("peeking multiple characters works",
	function()
		parser_context_instance.text = "string"

		char = parser_context_instance:peek(2)

		assert.equals("t", char)
		assert.equals(1, parser_context_instance.character_index)
	end)

	it("advancing works",
	function()
		parser_context_instance.text = "string"

		parser_context_instance:advance()

		assert.equals(2, parser_context_instance.character_index)
	end)

	it("advancing multplie characters works",
	function()
		parser_context_instance.text = "string"

		parser_context_instance:advance(2)

		assert.equals(3, parser_context_instance.character_index)
	end)

	it("rewinding multplie characters works",
	function()
		parser_context_instance.text = "string"

		parser_context_instance:advance(5)
		parser_context_instance:rewind(3)

		assert.equals(3, parser_context_instance.character_index)
	end)

	it("rewinding a character works",
	function()
		parser_context_instance.text = "string"

		parser_context_instance:advance(5)
		parser_context_instance:rewind(1)

		assert.equals(5, parser_context_instance.character_index)
	end)

	it("skip_until() works",
	function()
		parser_context_instance.text = "string"

		parser_context_instance:skip_until(
		{
			["i"] = true
		})

		assert.equals(4, parser_context_instance.character_index)
	end)

	it("skip_until() with inverse flag works",
	function()
		parser_context_instance.text = "    string"

		parser_context_instance:skip_until(
		{
			[" "] = true
		}, true)

		assert.equals(5, parser_context_instance.character_index)
	end)

	it("skip_until() does nothing at the end of the string",
	function()
		parser_context_instance.text = "string"
		parser_context_instance.character_index = 7

		parser_context_instance:skip_until(
		{
			[" "] = true
		}, true)

		assert.equals(7, parser_context_instance.character_index)
	end)

	it("skip_until() goes to the end of the string when no terminator is found",
	function()
		parser_context_instance.text = "string"

		parser_context_instance:skip_until(
		{
			[" "] = true
		})

		assert.equals(7, parser_context_instance.character_index)
	end)

	it("skip_until() with inverse flag does nothing when the current character isnt in the set",
	function()
		parser_context_instance.text = "string"
		parser_context_instance.character_index = 1

		parser_context_instance:skip_until(
		{
			[" "] = true
		}, true)

		assert.equals(1, parser_context_instance.character_index)
	end)

	it("match() correctly matches a single terminator",
	function()
		parser_context_instance.text = "terminator"

		local matched, terminator = parser_context_instance:match(helpers.create_trie({"terminator"}))

		assert.equals("terminator", terminator)
		assert.equals(true, matched)
		assert.equals(1, parser_context_instance.character_index)
	end)

	it("match() picks the longest matching terminator",
	function()
		parser_context_instance.text = "terminator"

		local matched, terminator = parser_context_instance:match(helpers.create_trie({"term", "terminator"}))

		assert.equals("terminator", terminator)
		assert.equals(true, matched)

		assert.equals(1, parser_context_instance.character_index)
	end)

	it("match() consumes the longest matching terminator when given the consume flag",
	function()
		parser_context_instance.text = "terminator string"

		local matched, terminator = parser_context_instance:match(helpers.create_trie({"term", "terminator"}), true)

		assert.equals("terminator", terminator)
		assert.equals(true, matched)

		assert.equals(11, parser_context_instance.character_index)
	end)

	it("match() returns false when no terminators match",
	function()
		parser_context_instance.text = "string"

		local matched, terminator = parser_context_instance:match(helpers.create_trie({"term", "terminator"}))

		assert.equals(nil, terminator)
		assert.equals(false, matched)
	end)

	it("match() returns false terminator_list is empty",
	function()
		parser_context_instance.text = "string"

		local matched, terminator = parser_context_instance:match(helpers.create_trie({}))

		assert.equals(nil, terminator)
		assert.equals(false, matched)
	end)

	it("match() ends parsing correctly when matching a terminator that is a prefix of another terminator",
	function()
		parser_context_instance.text = "|"

		local matched, terminator = parser_context_instance:match(helpers.create_trie({"|", "||"}), true)

		assert.equals("|", terminator)
		assert.equals(true, matched)

		assert.equals(2, parser_context_instance.character_index)
	end)

	it("is_EOF() returns false when not at the end of string",
	function()
		parser_context_instance.text = "string"

		assert.equals(false, parser_context_instance:is_EOF())
	end)

	it("is_EOF() returns false when at the last character",
	function()
		parser_context_instance.text = "string"
		parser_context_instance.character_index = 6

		assert.equals(false, parser_context_instance:is_EOF())
	end)

	it("is_EOF() returns true when at the end of the string",
	function()
		parser_context_instance.text = "string"
		parser_context_instance.character_index = 7

		assert.equals(true, parser_context_instance:is_EOF())
	end)

	describe("all members are correctly set by new(): ",
	function()
		local definition_table

		before_each(
		function()
			definition_table =
			{
				text = "string",
				character_index = 2
			}

			parser_context_instance = ParserContext:new(definition_table)
		end)

		it("text",
		function()
			assert.equals("string", parser_context_instance.text)
		end)

		it("character_index",
		function()
			assert.equals(2, parser_context_instance.character_index)
		end)

	end)

end)
