-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests.busted_definitions")

describe("Range class: ",
function()
	local Range = require("range")
	local range_instance

	before_each(
	function()
		range_instance = Range:new()
	end)

	it("contains() returns true when the number is inside the range",
	function()
		range_instance.start = 1
		range_instance.finish = 10

		assert.equals(true, range_instance:contains(5))
	end)

	it("contains() returns false when the number is outside the range",
	function()
		range_instance.start = 1
		range_instance.finish = 10

		assert.equals(false, range_instance:contains(-1))
	end)

	it("contains() returns true when one of the ends is infinite",
	function()
		range_instance.start = 1
		range_instance.finish = math.huge

		assert.equals(true, range_instance:contains(100))
	end)

	it("contains() returns true when the range is infinite",
	function()
		range_instance.start = -math.huge
		range_instance.finish = math.huge

		assert.equals(true, range_instance:contains(100))
	end)

	it("contains() works with inclusive ranges",
	function()
		range_instance.start = 1
		range_instance.start_inclusive = true
		range_instance.finish = 10
		range_instance.finish_inclusive = true

		assert.equals(true, range_instance:contains(10))
	end)

	it("contains() works with exclusive ranges",
	function()
		range_instance.start = 1
		range_instance.start_inclusive = false
		range_instance.finish = 10
		range_instance.finish_inclusive = false

		assert.equals(false, range_instance:contains(10))
	end)

	it("is_infinite() returns false when the range isnt infinite",
	function()
		range_instance.start = 1
		range_instance.finish = 10

		assert.equals(false, range_instance:is_infinite())
	end)

	it("is_infinite() returns true when the range is infinite",
	function()
		range_instance.start = 1
		range_instance.finish = math.huge

		assert.equals(true, range_instance:is_infinite())
	end)

	it("iterator() every element",
	function()
		range_instance.start = 1
		range_instance.start_inclusive = true
		range_instance.finish = 10
		range_instance.finish_inclusive = true

		local output = 0

		for i in range_instance:iterator() do
			output = output + i
		end

		assert.equals(55, output)
	end)

	it("iterator() with exclusive ends",
	function()
		range_instance.start = 1
		range_instance.start_inclusive = false
		range_instance.finish = 10
		range_instance.finish_inclusive = false

		local output = 0

		for i in range_instance:iterator() do
			output = output + i
		end

		assert.equals(44, output)
	end)

	it("iterator() when the range is infinite",
	function()
		range_instance.start = 1
		range_instance.finish = math.huge

		local output = 0

		for i in range_instance:iterator() do
			output = output + i
		end

		assert.equals(0, output)
	end)

	it("iterator() only iterates over integer numbers",
	function()
		range_instance.start = 0.5
		range_instance.finish = 10.5

		local output = 0

		for i in range_instance:iterator() do
			output = output + i
		end

		assert.equals(55, output)
	end)

	it("get_integer_boundries() respects inclusivity",
	function()
		range_instance.start = 1
		range_instance.start_inclusive = false
		range_instance.finish = 5
		range_instance.finish_inclusive = false

		local start, finish = range_instance:get_integer_boundries()

		assert.equals(2, start)
		assert.equals(4, finish)
	end)

	describe("members are correctly set by new(): ",
	function()
		local definition_table =
		{
			start = 1,
			start_inclusive = false,
			finish = 5,
			finish_inclusive = false,
		}

		before_each(
		function()
			range_instance = Range:new(definition_table)
		end)

		it("start",
		function()
			assert.equals(definition_table.start, range_instance.start)
		end)

		it("start_inclusive",
		function()
			assert.equals(definition_table.start_inclusive, range_instance.start_inclusive)
		end)

		it("finish",
		function()
			assert.equals(definition_table.finish, range_instance.finish)
		end)

		it("finish_inclusive",
		function()
			assert.equals(definition_table.finish_inclusive, range_instance.finish_inclusive)
		end)
	end)
end)
