-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local DatatypeValidator = require("datatype_validator")

DatatypeValidator.register_type("string",
{
	get_type = function(value)
		return type(value) == "string"
	end,

	iterator = function(value)
		local i = 0
		return function()
			if i < #value  then
				i = i + 1
				return value:sub(i, i)
			end
		end
	end,

	to_bool = function(value)
		return #value > 0
	end,

	conversions =
	{
		["number"] = function(value)
			return tonumber(value)
		end,

		["time"] = function(value)
			return helpers.parse_time(value)
		end,
	},

	bulk_conversions =
	{
		["number"] = function(t)
			local new_t = {}

			local j = 1
			local val
			for i = 1, #t do
				val = tonumber(t[i])
				if val then
					new_t[j] = val
					j = j + 1
				end
			end

			return new_t
		end,

		["time"] = function(t)
			local new_t = {}

			local j = 1
			local val
			for i = 1, #t do
				val = helpers.parse_time(t[i])
				if val then
					new_t[j] = val
					j = j + 1
				end
			end

			return new_t
		end,
	},
})

DatatypeValidator.register_type("boolean",
{
	get_type = function(value)
		return type(value) == "boolean"
	end,

	to_bool = function(value)
		return value
	end,

	conversions =
	{
		["number"] = function(value)
			return value and 1 or 0
		end,

		["string"] = function(value)
			return value and "true" or "false"
		end,
	},

	bulk_conversions =
	{
		["number"] = function(t)
			local new_t = {}

			for i = 1, #t do
				new_t[i] = t[i] and 1 or 0
			end

			return new_t
		end,

		["string"] = function(t)
			local new_t = {}

			for i = 1, #t do
				new_t[i] = t[i] and "true" or "false"
			end

			return new_t
		end,
	},
})

DatatypeValidator.register_type("number",
{
	get_type = function(value)
		return type(value) == "number"
	end,

	to_bool = function(value)
		return value ~= 0
	end,

	iterator = function(value)
		local i = 0
		return function()
			if i < value then
				i = i + 1
				return i
			end
		end
	end,

	conversions =
	{
		["string"] = function(value)
			return tostring(value)
		end,

		["time"] = function(value)
			return value
		end,

		["boolean"] = function()
			return true
		end,
	},

	bulk_conversions =
	{
		["string"] = function(t)
			local new_t = {}

			for i = 1, #t do
				new_t[i] = tostring(t[i])
			end

			return new_t
		end,

		["time"] = function(t)
			local new_t = {}

			for i = 1, #t do
				new_t[i] = t[i]
			end

			return new_t
		end,

		["boolean"] = function(t)
			local new_t = {}

			for i = 1, #t do
				new_t[i] = true
			end

			return new_t
		end,

	},
})

DatatypeValidator.register_type("time",
{
	get_type = function(value)
		return type(value) == "number"
	end,

	to_bool = function(value)
		return value ~= 0
	end,

	conversions =
	{
		["string"] = function(value)
			return helpers.time_to_string(value)
		end,

		["number"] = function(value)
			return value
		end,
	},

	bulk_conversions =
	{
		["string"] = function(t)
			local new_t = {}

			for i = 1, #t do
				new_t[i] = helpers.time_to_string(t[i])
			end

			return new_t
		end,

		["number"] = function(t)
			local new_t = {}

			for i = 1, #t do
				new_t[i] = t[i]
			end

			return new_t
		end,
	},
})

DatatypeValidator.register_type("range",
{
	get_type = function(value)
		return type(value) == "table" and value.instance_of == "Range"
	end,

	iterator = function(value)
		return value:iterator()
	end,

	to_bool = function()
		return true
	end,

	conversions =
	{
		["string"] = function(value)
			return tostring(value)
		end,
	},

	bulk_conversions =
	{
		["string"] = function(t)
			local new_t = {}

			for i = 1, #t do
				new_t[i] = tostring(t[i])
			end

			return new_t
		end,
	},
})

DatatypeValidator.register_type("table",
{
	iterator = function(value)
		local k
		local v
		return function()
			k, v = next(value, k)
			return v
		end
	end,

	get_type = function()
		return nil -- detecting tables is a special case thats hardcoded elsewhere
	end,

	to_bool = function(value)
		return #value > 0
	end,

	conversions =
	{
		["string"] = function(value)
			return tostring(value)
		end,
	},
})

DatatypeValidator.register_type("function",
{
	get_type = function(value)
		return type(value) == "function" or (type(value) == "table" and value.__call)
	end,

	to_bool = function()
		return true
	end,

	conversions =
	{
		["string"] = function(value)
			return tostring(value)
		end,
	},
})

