-- Copyright (C) 2025 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

lush.Command:new(
{
	name = "lsscripts",
	description = [[lists out all the scripts]],
	privs = {server = true},
	arguments = {},
	callback = function(ctx, args, options)
		local scripts = core.get_dir_list(string.format("%s/lush_scripts", core.get_worldpath()), false)

		ctx.stdout:concat(scripts)
	end,
})

lush.Command:new(
{
	name = "run",
	description = [[runs the specified lush script from the "lush_scripts" directory from the worldpath]],
	options = {["cache"] = false, ["absolute"] = false},
	privs = {server = true},
	arguments = 
	{
		{
			name = "lush_script",
			single = true
		}
	},
	callback = function(ctx, args, options)
		ctx:interrupt()
		local subctx = lush.ShellContext:new()
		local path = options["absolute"] and args.lush_script or string.format("%s/lush_scripts/%s", core.get_worldpath(), args.lush_script),
		subctx.env:inherit(ctx.env)

		lush.Interpreter.run_file(
			path,
			subctx, 
			{
				complete_callback = function(success, err_msg, subctx)
					ctx.stdout:concat(subctx.stdout)
					ctx.stderr = subctx.stderr
					ctx.exit_status = subctx.exit_status

					ctx:stop_interruption()
				end
			}
		)

	end,
})

lush.Command:new(
{
	name = "apply_ruleset",
	description = "converts a list into the format specified by the ruleset. this is meant for argument validation",
	arguments =
	{
		{
			name = "list",
			type = "table",
			single = true
		},
		{
			name = "ruleset",
			type = "table",
			single = true
		}
	},
	callback = function(ctx, args, options)
		local output = lush.DatatypeValidator.validate(args.list, args.ruleset, ctx)

		ctx.stdout:push(output)
	end,
})

lush.Command:new(
{
	name = "eval_command_args",
	description = "returns the args that are passed to a command for debug purposes, dosent work well with options",
	arguments = 
	{
		{
			name = "command_name",
			type = "string",
			single = true
		},
		{
			rule_type = "block",
			name = "arg",
		}
	},
	callback = function(ctx, args, options)
		local output = lush.DatatypeValidator.validate(args.arg, lush.RegisteredCommands[args.command_name].arguments, ctx)

		ctx.stdout:push(output)
	end,
})

local subshell_options = {share_environment = true}

lush.Command:new(
{
	name = "lush",
	description = "runs a shell",
	arguments =
	{
		{
			name = "code",
			type = "string",
			single = true,
		}
	},
	callback = function(ctx, args, options)
		local subshell_ctx = lush.ShellContext:new({env = ctx.env})
		function complete_callback(success, err_msg, subshell_ctx)
			ctx.stdout = subshell_ctx.stdout
			ctx.stderr = subshell_ctx.stderr
			ctx.exit_status = subshell_ctx.exit_status

			ctx:stop_interruption()
		end

		subshell_options.complete_callback = complete_callback

		ctx:interrupt()
		lush.Interpreter.run_shell(args.code, subshell_ctx, subshell_options)

	end,
})

lush.Command:new(
{
	name = "parse",
	description = "parses a shell, returns the parse tree that can be used by the eval command",
	arguments =
	{
		{
			name = "code",
			type = "string",
			single = true,
		}
	},
	callback = function(ctx, args, options)
		local success, error_message, parse_tree = lush.Interpreter.parse_shell(args.code)

		if not success then
			error(error_message)
		end

		ctx.stdout:push(parse_tree)
	end,
})

lush.Command:new(
{
	name = "dump_parse_tree",
	description = "Dumps the parse tree as a human readable string",
	options = {nocolor = false},
	arguments = {},
	callback = function(ctx, args, options)
		local dump_ctx = lush.DumpContext:new()

		if options.nocolor then
			dump_ctx.use_color = false
		end

		ctx.stdout:push(lush.helpers.dumpIR(ctx.stdin:get(1), dump_ctx))
	end,
})

lush.Command:new(
{
	name = "dump_function",
	description = "Dumps a function written in lush as a human readable string",
	options = {nocolor = false},
	arguments =
	{
		{
			name = "func",
			type = "function",
			single = true
		}
	},
	callback = function(ctx, args, options)
		if type(args.func) ~= "table" or args.func.instance_of ~= "LanguageConstruct_FunctionDefinition" then
			error("Function can't be dumped because it isn't a lush defined function")
		end
		local dump_ctx = lush.DumpContext:new()

		if options.nocolor then
			dump_ctx.use_color = false
		end

		ctx.stdout:push(lush.helpers.dumpIR(args.func, dump_ctx))
	end,
})

lush.Command:new(
{
	name = "eval",
	description = "evaluates the output of the parse command",
	arguments = {},
	stdin_type = "table",
	callback = function(ctx, args, options)
		local subshell_ctx = lush.ShellContext:new({env = ctx.env})
		ctx:interrupt()

		ctx.stdin:get(1):evaluate(subshell_ctx,
		{
			share_environmen = true,
			complete_callback = function(subshell_ctx)
				ctx.stdout = subshell_ctx.stdout
				ctx.stderr = subshell_ctx.stderr
				ctx.exit_status = subshell_ctx.exit_status

				ctx:stop_interruption()
			end
		})
	end,
})

